package nanorep.nanowidget.DataClass

import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.net.Uri
import android.util.Log
import android.util.LruCache
import com.nanorep.nanoclient.Connection.NRConnection
import com.nanorep.nanoclient.Connection.NRError
import com.nanorep.nanoclient.Connection.NRErrorCodes
import com.nanorep.nanoclient.Connection.SupportDownloader
import com.nanorep.nanoclient.Interfaces.NRQueryResult
import com.nanorep.nanoclient.Nanorep
import com.nanorep.nanoclient.Response.NRFAQGroupItem
import com.nanorep.nanoclient.exception.NRConnectionException
import com.nanorep.nanoclient.model.NRResult
import com.nanorep.nanoclient.model.ResultResponse
import com.nanorep.nanoclient.model.SupportLabel
import com.nanorep.nanoclient.network.OnDataResponse
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.GlobalScope
import kotlinx.coroutines.launch
import nanorep.nanowidget.Fragments.ResultsFragment
import java.util.*

const val InitialLabelsKey: String = "-1"

class FAQDataSource {

    // An updated list of the current data
    private var currentList: MutableList<ResultResponse> = mutableListOf()

    // An LRU cache object that saves the fetched labels and the bitmaps
    val cache = LruCache<String, Any>(1024 * 1024 * 8)

    // A function used to notify on a listUpdate
    lateinit var notifyDataSetChanged: (List<ResultResponse>?) -> Unit

    // A function used to notify on a specific item changes
    lateinit var notifyItemChanged: (Int) -> Unit

    fun setLabel(labelId: String?, updateList: (List<ResultResponse>?) -> Unit, notifyItem: (Int) -> Unit) {

        this.notifyDataSetChanged = updateList
        this.notifyItemChanged = notifyItem

        labelId?.run {

            // Check if the label has been already cached
            (cache.get(labelId) as? SupportLabel)?.run {

                fetchData(this)

            } ?: kotlin.run {

                // Finds the label at the currentList
                (currentList.find { it.responseId == labelId } as? SupportLabel)?.run {

                    fetchData(this)

                }
            }

        } ?: kotlin.run {

            // Initial Labels handling
            handleInitialState()
        }
    }

    private fun handleInitialState() {

        // If the initial labels have been already cached:
        (cache.get(InitialLabelsKey) as? SupportLabel)?.run {

            fetchData(this)

        } ?: kotlin.run {

            // Download the SupportCenter's initial labels
            GlobalScope.launch(Dispatchers.IO) {

                Nanorep.getInstance().getSupportCenterLabels { downloadedLabels ->

                    val initialLabel = SupportLabel().apply {
                        id = InitialLabelsKey
                    }

                    initialLabel.children = downloadedLabels.filter { it.isSupportCenterLabel }

                    fetchData(initialLabel)

                }
            }
        }
    }

    /**
     * Fetches the selected label's data.
     * Updates the current List with it's sub labels and faqs
     * Adds the label to the cache
     */
    private fun fetchData(selectedLabel: SupportLabel?) {

        selectedLabel?.let { label ->

            if ( cache.get(label.id) == null ) {
                cache.put(label.id, label)
            }

            label.children?.run { fetchLabels(this) }

            fetchFaqs(label){ fetchedFaqs ->

                fetchedFaqs.run {

                    label.faqs = this

                    // Updates the label's faqs at the cache
                    cache.put(label.id, label)

                    updateList(this)
                    Log.d("FAQDataSource","$size faqs fetched")
                }
            }
        }
    }

    private fun fetchLabels(labelList: List<SupportLabel>) {

        currentList.clear()

        labelList.run {

            forEach {
                if (cache.get(it.id) == null) {
                    cache.put(it.id, it)
                }
            }

            updateList(this)
            Log.d("FAQDataSource","$size labels fetched")

            fetchLabelsIcons(currentList)
        }
    }

    private fun fetchFaqs (supportLabel: SupportLabel, onFetched: (MutableList<ResultResponse>) -> Unit) {

        supportLabel.run {

            // Checks if the faqs have been already fetched for this label
            faqs?.run {

                onFetched.invoke(this)

            } ?: kotlin.run {

                Nanorep.getInstance().updateFaqList(Integer.toHexString(id.toInt()), null, FaqResponse(onFetched))

            }

        }
    }

    private fun fetchLabelsIcons(labels: List<ResultResponse>) {

        labels.forEachIndexed { index, label ->

            (label as? SupportLabel)?.run {

               getIconSrc()?.let { url ->

                   // Trying to get the icon from the cache
                   cache.get(url)?.run {
                       (this as Bitmap).run { bitmap = this }

                   } ?: kotlin.run {

                       // If the icon is not cached, download it:
                       downloadIcon(url, index)
                   }
               }
            }
        }
    }

    private fun SupportLabel.downloadIcon(url: String, index: Int) {

        getIconSrc()?.takeIf { it.isNotEmpty() }?.let { iconUrl ->

            SupportDownloader.download(Uri.parse(iconUrl),

                    object : OnDataResponse<Bitmap>() {

                        override fun onSuccess(response: Bitmap?) {
                            bitmap = response
                            cache.put(url, bitmap)
                            notifyItemChanged.invoke(index)
                        }

                        override fun onError(error: NRConnectionException?) {
                            Log.d("FAQDataSource", "Error downloading icon for label $name")
                            Nanorep.getInstance().widgetListener.onError(NRError.error(ResultsFragment.TAG, NRErrorCodes.CONNECTION, error?.message))
                        }

                    }) {

                BitmapFactory.decodeByteArray(it, 0, it.size)
            }
        }
    }

    private fun updateList(resultResponses: List<ResultResponse>) {
        currentList.addAll(resultResponses)
        notifyDataSetChanged(resultResponses)
    }

    internal class FaqResponse(private val onFetched: (MutableList<ResultResponse>) -> Unit?) : NRConnection.Listener{

        override fun response(responseParam: Any?, status: Int, error: NRError?) {

            error?.run {
                Nanorep.getInstance().widgetListener.onError(error)
                return
            }

            (responseParam as? HashMap<String, Any>)?.run {
                generateSupportResultsArray(NRFAQGroupItem(this).answers)?.run {

                    onFetched.invoke(this)

                    Log.d("FAQDataSource","$size faqs fetched")
                }
            }
        }

        override fun log(tag: String, msg: String) {
            Log.d("FAQDataSource","faqs request url: $msg")
        }

        private fun generateSupportResultsArray(queryResults: ArrayList<NRQueryResult>?): MutableList<ResultResponse>? {
            queryResults?.run {
                val results = ArrayList<ResultResponse>()
                for (result in this) {
                    results.add(NRResult(result, NRResult.RowType.TITLE))
                }
                return results
            }
            return null
        }
    }

    fun onDestroy() {
        cache.evictAll()
    }
}