package nanorep.nanowidget;

import android.app.Dialog;
import android.content.Context;
import android.content.DialogInterface;
import android.os.Build;
import android.support.annotation.NonNull;
import android.support.annotation.RequiresApi;
import android.support.v7.app.AlertDialog;
import android.support.v7.widget.DividerItemDecoration;
import android.support.v7.widget.LinearLayoutManager;
import android.support.v7.widget.RecyclerView;
import android.util.Log;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.View;
import android.view.inputmethod.InputMethodManager;
import android.widget.TextView;

import com.nanorep.nanoclient.Connection.NRConnection;
import com.nanorep.nanoclient.Connection.NRError;
import com.nanorep.nanoclient.Nanorep;
import com.nanorep.nanoclient.Response.NRFAQGroupItem;
import com.nanorep.nanoclient.exception.NRConnectionException;
import com.nanorep.nanoclient.model.ContextValue;
import com.nanorep.nanoclient.model.DefaultResponse;
import com.nanorep.nanoclient.model.NRLabel;
import com.nanorep.nanoclient.network.OnDataResponse;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import nanorep.nanowidget.Components.ContextSelectionAdapter;
import nanorep.nanowidget.DataClass.ContextSelectionHandler;
import nanorep.nanowidget.DataClass.NRConfigFetcherListener;
import nanorep.nanowidget.DataClass.NRFetchedDataManager;
import nanorep.nanowidget.DataClass.SearchFetcherListener;
import nanorep.nanowidget.DataClass.NRResult;

import static com.nanorep.nanoclient.Connection.NRErrorCodes.CONNECTION;

public class NanorepUI {

    public interface DataManagerErrorListener {

        void onError(NRError error);
    }

    public interface DataListener extends DataManagerErrorListener {

        void insertRows(ArrayList<NRFAQGroupItem> groups);
    }

    public static String NoResults = "noResults";
    private String TAG = getClass().getName();

    private NRFetchedDataManager dataManager;
    private List<NRResult> faqs;
    private HashMap<String, NRResult> articlesStack = new HashMap<>();
    private NRLabel label;
    private Context context;
    private Nanorep.NanoRepWidgetListener widgetListener;
    private AlertDialog contextSelectionDialog;
    private SearchFetcherListener fetcherListener;
    private String searchContextValue;
    private DataManagerErrorListener dataManagerErrorListener;
    private static NanorepUI instance;

    public void setContext(Context context) {
        this.context = context;
        if (dataManager == null) {
            initDataManager();
        }
    }

    public void setDataManagerErrorListener(DataManagerErrorListener dataManagerErrorListener) {
        this.dataManagerErrorListener = dataManagerErrorListener;
    }

    public void clearInstance() {
        context = null;
        dataManager = null;
        widgetListener = null;
        fetcherListener = null;
        dataManagerErrorListener = null;
        instance = null;
    }

    private void initDataManager() {

        if (context != null) {
            dataManager = new NRFetchedDataManager(context, new NRConfigFetcherListener() {

                @RequiresApi(api = Build.VERSION_CODES.LOLLIPOP)
                @Override
                public void onConfigurationReady() {
                }

                @Override
                public void insertRows(ArrayList<NRFAQGroupItem> groups) {

                }

                @Override
                public void onError(NRError error) {

                    if (dataManagerErrorListener != null) {
                        dataManagerErrorListener.onError(error);
                    }

                    if (widgetListener == null && Nanorep.getInstance() != null) {
                        widgetListener = Nanorep.getInstance().getWidgetListener();
                    }

                    if (widgetListener != null) {
                        Nanorep.getInstance().getWidgetListener().onError(error);
                    }
                }

                @Override
                public void noResultFound(NRResult result, String searchContext) {
                    List<NRResult> results = new ArrayList<>();
                    results.add(result);
                    dataManager.getFetcherListener().insertRows(results, searchContext);
                    if (searchContextValue == null) {
                        clearContextFromSession();
                    }
                }

                @Override
                public void showContextSelectionDialog(ContextValue contextValues) {
                    presentContextDialog(contextValues);
                }
            });
        }

    }

    public void setSearchContextValue(String searchContextValue) {
        this.searchContextValue = searchContextValue;
    }

    public String getSearchContextValue() {
        return searchContextValue;
    }

    public synchronized static NanorepUI getInstance() {
        if (instance == null) {
            instance = new NanorepUI();
        }
        return instance;
    }

    public NRFetchedDataManager getDataManager(SearchFetcherListener fetcherListener) {
        if (dataManager == null) {
            initDataManager();
        }
        this.fetcherListener = fetcherListener;
        if (dataManager != null) {
            dataManager.setFetcherListener(fetcherListener);
        }
        return dataManager;
    }

    public NRFetchedDataManager getDataManager() {
        return dataManager;
    }

    public List<NRResult> getResults() {
        return faqs;
    }

    public void setResults(List<NRResult> faqs) {
        this.faqs = faqs;
    }

    public void addToArticleStuck(NRResult result) {
        if (result != null && result.getFetchedResult() != null) {
            articlesStack.put(result.getFetchedResult().getId(), result);
        } else {
            articlesStack.put(NoResults, result);
        }
        Log.d("","");
    }

    public void removeFromArticleStack(String resultID) {
        articlesStack.remove(resultID);
    }

    public NRResult getFromArticleStack(String resultID) {
        return articlesStack.get(resultID);
    }

    public void setLabel(NRLabel label) {
        this.label = label;
        if (label != null) {
            String labelContext = label.getContext();
            if (labelContext != null && !labelContext.isEmpty()) {
                searchContextValue = label.getContext();
            }
        }
    }

    public void updateLabelContext() {
        if (label != null) {
            updateSessionContext(label.getContext());
        }
    }

    public void clearCurrentLabel() {
        setLabel(null);
        clearAllContexts();
    }

    public NRLabel getLabel() {
        return label;
    }

    private void presentContextDialog(ContextValue contextValues) {
        if (context != null) {
            View contextSelectionDialogView = LayoutInflater.from(context).inflate(R.layout.context_selection_dialog, null);
            final TextView contextTitle = contextSelectionDialogView.findViewById(R.id.context_selection_title);
            final TextView cancelButton = contextSelectionDialogView.findViewById(R.id.cancelButton);
            RecyclerView recycler = contextSelectionDialogView.findViewById(R.id.context_selection_recycler);
            recycler.setLayoutManager(new LinearLayoutManager(contextSelectionDialogView.getContext()));
            recycler.addItemDecoration(new DividerItemDecoration(contextSelectionDialogView.getContext(), DividerItemDecoration.VERTICAL));

            final ContextSelectionAdapter adapter = new ContextSelectionAdapter(contextValues);
            ContextSelectionHandler contextSelectionHandler = new ContextSelectionHandler(contextValues, new ContextSelectionHandler.ContextSelectionListener() {

                @Override
                public void updateContextData(@org.jetbrains.annotations.NotNull ContextValue contextValues) {
                    contextTitle.setText(R.string.contextDialogTitle);
                    contextTitle.append(" " + contextValues.getContextKey());
                    adapter.setContextList(contextValues);
                    adapter.notifyDataSetChanged();
                }

                @Override
                public void onContextSelected(@NonNull String searchContext) {
                    if (!searchContext.isEmpty()) {
                        if (searchContextValue != null) {
                            // adds the dynamic selected context to the current label context
                            searchContextValue += "," + searchContext;
                        } else {
                            searchContextValue = searchContext;
                        }
                    }
                    dataManager.presentResultsForSelectedContext(searchContextValue);
                    contextSelectionDialog.dismiss();
                }
            });

            adapter.setContextListener(contextSelectionHandler);
            recycler.setAdapter(adapter);

            cancelButton.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View view) {

                    fetcherListener.onContextDialogCanceled();

                    contextSelectionDialog.dismiss();
                }
            });

            contextTitle.setText(R.string.contextDialogTitle);
            contextTitle.append(" " + contextValues.getContextKey());

            AlertDialog.Builder builder = new AlertDialog.Builder(contextSelectionDialogView.getContext());
            contextSelectionDialog = builder.setView(contextSelectionDialogView).create();
            contextSelectionDialog.setCanceledOnTouchOutside(false);
            contextSelectionDialog.setOnKeyListener(new Dialog.OnKeyListener() {
                @Override
                public boolean onKey(DialogInterface arg0, int keyCode,
                                     KeyEvent event) {
                    if (keyCode == KeyEvent.KEYCODE_BACK) {
                        cancelButton.callOnClick();
                        fetcherListener.onContextDialogCanceled();
                        return true;
                    }
                    return true;
                }
            });

            contextSelectionDialog.show();
        }
    }

    public void hideKeyboard(View view) {
        if (view != null) {
            Context context = view.getContext();
            if (context != null) {
                InputMethodManager imm = (InputMethodManager) context.getSystemService(Context.INPUT_METHOD_SERVICE);
                if (imm != null) {
                    imm.hideSoftInputFromWindow(view.getWindowToken(), 0);
                }
            }
        }
    }

    public void updateDynamicContextValue(String updatedContext) {

        if ((updatedContext == null || updatedContext.isEmpty()) && label != null) {
            updateSessionContext(label.getContext());

        } else if (label == null && updatedContext == null){
            clearAllContexts();

        } else if (!updatedContext.equals(searchContextValue)) {

            StringBuilder builder = new StringBuilder();

            if (label != null && !label.getContext().equals(updatedContext)) {
                updatedContext = label.getContext() + ',' + updatedContext;
            }

            for (String context : updatedContext.split(",")) {
                String[] ctx = context.split(":");
                if (ctx.length != 2) {
                    return;
                }
                if (builder.length() > 0) {
                    builder.append(", ");
                }
                builder.append(ctx[1]);
            }

            setSearchContextValue(builder.toString());
        } else {
            setSearchContextValue(updatedContext);
        }
    }

    private void updateSessionContext(final String sessionContext) {

        Nanorep.getInstance().setContext(null, new OnDataResponse<DefaultResponse>() {
            @Override
            public void onSuccess(DefaultResponse response) {
                setSearchContextValue(null);
                Log.d("ContextLog", "Context cleared successfully");
                if (dataManager != null) {
                    dataManager.clearSearchContext();
                }
                searchContextValue = null;
                if (sessionContext != null) {
                    Nanorep.getInstance().setContext(sessionContext, new OnDataResponse<DefaultResponse>() {
                        @Override
                        public void onSuccess(DefaultResponse response) {
                            setSearchContextValue(sessionContext);
                            Log.d("ContextLog", "Context has been updated successfully");
                        }

                        @Override
                        public void onError(NRConnectionException error) {
                            if (widgetListener != null) {
                                Nanorep.getInstance().getWidgetListener().onError(NRError.error(TAG, CONNECTION, "Error on update Context"));
                            }
                        }
                    });
                }
            }

            @Override
            public void onError(NRConnectionException error) {
                if (widgetListener != null) {
                    Nanorep.getInstance().getWidgetListener().onError(NRError.error(TAG, CONNECTION, "Error on clearing Context"));
                }
            }
        });
    }

    
    public void clearAllContexts() {

        label = null;

        if (dataManager != null) {
            dataManager.clearSearchContext();
        }

        clearContextFromSession();

    }

    private void clearContextFromSession() {
        Nanorep.getInstance().setContext(null, new OnDataResponse<DefaultResponse>() {
            @Override
            public void onSuccess(DefaultResponse response) {
                setSearchContextValue(null);
                Log.d("ContextLog", "Context cleared successfully");
            }

            @Override
            public void onError(NRConnectionException error) {
                Nanorep.getInstance().getWidgetListener().onError(NRError.error(TAG, CONNECTION, "Error on update Context"));
            }
        });
    }

    public void prepareDataSource(final DataListener dataListener) {
        if (Nanorep.getInstance().getAccountParams().isLabelsMode() && Nanorep.getInstance().getNRConfiguration().getLabels() != null) {
            dataListener.insertRows(null);
            return;
        }

        NRConnection.Listener listener = new NRConnection.Listener() {
            @Override
            public void response(Object responseParam, int status, NRError error) {

                if (error != null) {
                    dataListener.onError(error);
                } else {
                    final ArrayList<NRFAQGroupItem> groups = new ArrayList<>();
                    if (responseParam instanceof HashMap) {
                        final HashMap<String, Object> map = (HashMap<String, Object>) responseParam;
                        final NRFAQGroupItem group = new NRFAQGroupItem(map);
                        groups.add(group);
                        dataListener.insertRows(groups);
                    }
                }
            }

            @Override
            public void log(String tag, String msg) {

            }
        };

        Nanorep.getInstance().updateFaqList(null, Nanorep.getInstance().getAccountParams().getContext(), listener);
    }

    public interface FaqsListener {
        void onFaqsFetched(ArrayList<NRFAQGroupItem> groups);
    }

}
