package nanorep.nanowidget.Components;

import android.content.Context;
import android.graphics.Color;
import android.text.Editable;
import android.text.TextWatcher;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.View;
import android.view.inputmethod.EditorInfo;
import android.widget.ImageButton;
import android.widget.LinearLayout;
import android.widget.TextView;

import nanorep.nanowidget.Components.AbstractViews.NRCustomSearchBarView;
import nanorep.nanowidget.R;
import nanorep.nanowidget.SearchViewsProvider;
import nanorep.nanowidget.interfaces.NRSearchBarListener;


/**
 * Created by nissimpardo on 07/06/16.
 */
public class NRSearchBar extends NRCustomSearchBarView implements TextWatcher, TextView.OnEditorActionListener {

    private final int searchIconResourceId = R.drawable.ic_search_black_24dp;
    private final int closeIconResourceId = R.drawable.ic_close_black_24dp;
    private NRSearchBarListener mListener;
    private ImageButton mSearchButton;
    private TextView labelName;
    private LinearLayout searchBarLayout;
    private LinearLayout labelLayout;
    private SearchViewsProvider viewsProvider;
    private String searchQuery;

    public NRSearchBar(Context context) {
        super(context);
        LayoutInflater.from(context).inflate(R.layout.search_bar, this);
    }

    @Override
    public void onViewAdded(View child) {
        super.onViewAdded(child);
        searchBarLayout = child.findViewById(R.id.searchBarLayout);
        mSearchEditText = child.findViewById(R.id.searchText);
        labelLayout = child.findViewById(R.id.labelLayout);
        labelName = child.findViewById(R.id.search_bar_label_name_textView);


        mSearchEditText.addTextChangedListener(this);
        mSearchEditText.setOnEditorActionListener(this);
        mSearchButton = child.findViewById(R.id.searchButton);
        mSearchButton.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View view) {
                if (mSearchEditText.getText().length() > 0) {
                    mSearchButton.setImageResource(searchIconResourceId);
                    clear();
                }
            }
        });

        mSearchEditText.setOnKeyListener(new OnKeyListener() {
            @Override
            public boolean onKey(View v, int keyCode, KeyEvent event) {
                if (event.getAction() == KeyEvent.ACTION_UP) {
                    if (keyCode == KeyEvent.KEYCODE_BACK) {
                        clear();
                        return true;
                    }
                }
                return false;
            }
        });
    }

    public void setListener(NRSearchBarListener listener) {
        mListener = listener;
    }


    private int resId(String resName) {
        return getResources().getIdentifier(resName, "drawable", getContext().getPackageName());
    }

    public void updateText(String text) {
        mSearchEditText.clearFocus();
        mSearchEditText.setText(text);
    }

    @Override
    public void clearText() {
        updateText("");
    }


    /**
     *  @param displayLabel = false : Shows the search layout
     * @param labelTitle = if (show) this is the Label text. Else, it is the search text. Being ignored if empty
     */

    public void adaptSearchBarDisplay(boolean displayLabel, String labelTitle) {
        if (displayLabel && !viewsProvider.isSearchBarAlwaysOnTop() && (labelTitle != null && !labelTitle.isEmpty())) {
            searchBarLayout.setVisibility(GONE);
            labelName.setText(labelTitle);
            labelLayout.setVisibility(VISIBLE);
        } else {
            labelLayout.setVisibility(GONE);
            if (searchQuery != null && !searchQuery.isEmpty()) {
                mSearchEditText.setText(searchQuery);
            }
            searchBarLayout.setVisibility(VISIBLE);
        }
    }

    public void setHint(String text) {
        mSearchEditText.setHint(text);
    }

    @Override
    public void setViewsProvider(SearchViewsProvider viewsProvider) {
        this.viewsProvider = viewsProvider;
    }

    public String getText() {
        searchQuery = mSearchEditText.getText().toString();
        return searchQuery;
    }

    private void clear() {
        mSearchEditText.clearFocus();
        hideKeyboard();
        updateText("");
        searchQuery = "";
        mListener.onClearSearchText(true);
    }

    @Override
    public void beforeTextChanged(CharSequence charSequence, int i, int i1, int i2) {

    }

    @Override
    public void onTextChanged(CharSequence charSequence, int i, int i1, int i2) {
        boolean isSequenceEmpty = (charSequence == null || charSequence.length() == 0);
        mSearchButton.setImageResource(!isSequenceEmpty ? closeIconResourceId : searchIconResourceId);
        if (isSequenceEmpty) {
            hideKeyboard();
            mListener.clearAutoComplete();
        }
    }

    @Override
    public void afterTextChanged(Editable editable) {
        if (editable != null && editable.length() > 0) {
            mListener.fetchSuggestionsForText(mSearchEditText.getText().toString());
        } else {
            mListener.onEmptyQuery();
        }
    }

    @Override
    public boolean onEditorAction(TextView v, int actionId, KeyEvent event) {
        if (actionId == EditorInfo.IME_ACTION_SEARCH) {
            mListener.searchForText(mSearchEditText.getText().toString());
            hideKeyboard();
            return true;
        }
        return false;
    }

    @Override
    public void setBackgroundColor(int color) {
        super.setBackgroundColor(color);
        searchBarLayout.setBackgroundColor(Color.WHITE);
    }



}
