package com.nanorep.nanoclient;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.Typeface;
import android.net.Uri;
import android.os.Build;
import android.os.Handler;
import android.support.v4.app.FragmentManager;
import android.text.Spannable;
import android.text.SpannableStringBuilder;
import android.text.style.StyleSpan;
import android.util.Base64;
import android.util.Log;


import com.nanorep.nanoclient.Channeling.NRChanneling;
import com.nanorep.nanoclient.Connection.NRCacheManager;
import com.nanorep.nanoclient.Connection.NRConnection;
import com.nanorep.nanoclient.Connection.NRError;
import com.nanorep.nanoclient.Connection.NRUtilities;
import com.nanorep.nanoclient.Interfaces.NRExtraDataListener;
import com.nanorep.nanoclient.Interfaces.NRQueryResult;
import com.nanorep.nanoclient.Log.NRLogger;
import com.nanorep.nanoclient.RequestParams.NRFAQLikeParams;
import com.nanorep.nanoclient.RequestParams.NRSearchLikeParams;
import com.nanorep.nanoclient.Response.NRConfiguration;
import com.nanorep.nanoclient.Response.NRFAQAnswer;
import com.nanorep.nanoclient.Response.NRFAQGroupItem;
import com.nanorep.nanoclient.Response.NRSearchResponse;
import com.nanorep.nanoclient.Response.NRSuggestions;
import com.nanorep.nanoclient.exception.NRConnectionException;
import com.nanorep.nanoclient.model.DefaultResponse;
import com.nanorep.nanoclient.model.HelloResponse;
import com.nanorep.nanoclient.model.NRContextList;
import com.nanorep.nanoclient.network.OnDataResponse;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


/**
 * Created by nissimpardo on 07/08/2016.
 */
public class Nanorep {
    private AccountParams mAccountParams;
    private NRConfiguration mCnf;
    private NRLogger nrLogger;
    private Context mContext;

    private String mSessionId;
    private long mDelay;
    private HashMap<String, NRSearchResponse> mCachedSearches;
    private HashMap<String, NRSuggestions> mCachedSuggestions;
    private HashMap<String, ArrayList<OnFAQAnswerFetchedListener>> faqRequestListenersMap;
    private Handler mHandler;
    private NanoRepWidgetListener widgetListener;
    private FragmentManager fragmentManager;
    private String searchQuery;
    private boolean keepAliveRunning;

    public NanoRepWidgetListener getWidgetListener() {
        return widgetListener;
    }

    public String getSessionId() {
        return mSessionId;
    }

    public interface NanoRepWidgetListener {
        void personalInfoWithExtraData(String channelDescription, String extraData, NRExtraDataListener listener);
        void onSubmitSupportForm(String formData);
        void onConfigurationFetched();
        void onCachedImageRequest(String url, NRCachedImageResponder responder);
        void onInitializationFailure();
    }

    public interface NRCachedImageResponder {
        void onBitmapResponse(Bitmap bitmap);
    }

    private static Nanorep INSTANCE;

    public interface OnSearchResultsFetchedListener {
        void onSearchResponse(NRSearchResponse response, NRError error);
    }

    public interface OnSuggestionsFetchedListener {
        void onSuggestionsFetched(NRSuggestions suggestions, NRError error);
    }

    public interface OnLikeSentListener {
        void onLikeSent(boolean success);
    }

    public interface OnFAQAnswerFetchedListener {
        void onFAQAnswerFetched(NRFAQAnswer faqAnswer, NRError error);
    }

    public interface OnConfigurationFetchedListener {
        void onConfigurationFetched(NRError error);
    }

    public void setAccountParams(AccountParams accountParams) {
        this.mAccountParams = accountParams;
    }

    public void init(Context context, FragmentManager fragmentManager,
                     AccountParams accountParams, NanoRepWidgetListener widgetListener) {
        clearCache();
        this.widgetListener = widgetListener;
        this.mContext = context;
        this.fragmentManager = fragmentManager;
        if (getAccountParams() != null && accountParams.compareTo(getAccountParams()) != 0) {
            mCnf = null;
        }

        setAccountParams(accountParams);
        this.nrLogger = new NRLogger();
//        this.mCnf = null;

        fetchConfiguration();
    }

    public void reset() {
//        mCnf = null;
        mSessionId = null;
        if (mHandler != null) {
            mHandler.removeCallbacksAndMessages(null);
        }
    }

    public static Nanorep getInstance() {
        if (INSTANCE == null) {
            synchronized (Nanorep.class) {
                if (INSTANCE == null) {
                    INSTANCE = new Nanorep();
                }
            }
        }
        return INSTANCE;
    }

    private HashMap<String, NRSearchResponse> getCachedSearches() {
        if (mCachedSearches == null) {
            mCachedSearches = new HashMap<>();
        }
        return mCachedSearches;
    }

    private HashMap<String, NRSuggestions> getCachedSuggestions() {
         if (mCachedSuggestions == null) {
            mCachedSuggestions = new HashMap<>();
        }
        return mCachedSuggestions;
    }

    private HashMap<String, ArrayList<OnFAQAnswerFetchedListener>> getFaqRequestListenersMap() {
        if (faqRequestListenersMap == null) {
            faqRequestListenersMap = new HashMap<>();
        }
        return faqRequestListenersMap;
    }

    public void clearCache() {
        if (mCachedSearches != null) {
            Log.i("formData", "cache cleared");
            mCachedSearches.clear();
        }
    }

    private void callKeepAlive() {
        Uri.Builder uri = mAccountParams.getUri();
        uri.appendEncodedPath("api/widget/v1/keepAlive.js");
        uri.appendQueryParameter("sid",mSessionId);
        NRConnection.connection(uri, new OnDataResponse<String>() {
            @Override
            public void onSuccess(String response) {
                response.toString();
            }

            @Override
            public void onError(NRConnectionException error) {
                error.toString();
            }
        });
    }

    public void reportChanneling(String articleId, String type, String channelName) {
        Uri.Builder uri = mAccountParams.getUri();
        uri.appendEncodedPath("api/channeling/v1/report");
        uri.appendQueryParameter("account",mAccountParams.getAccount());
        uri.appendQueryParameter("kb",mAccountParams.getKnowledgeBase());
        uri.appendQueryParameter("articleId",articleId);
        uri.appendQueryParameter("trafficSource","FAQ");
        uri.appendQueryParameter("reportedEscalationType",type);
        uri.appendQueryParameter("apiName",channelName);

        NRConnection.connection(uri, new OnDataResponse<String>() {
            @Override
            public void onSuccess(String response) {
                response.toString();
            }

            @Override
            public void onError(NRConnectionException error) {

            }
        });
    }

    private void keepAlive(final long interval) {
        callKeepAlive();
        mHandler = new Handler();

        mHandler.postDelayed(new Runnable() {
            public void run() {
                keepAlive(interval);
            }
        }, interval);
    }

    private void executeRequest(final Uri.Builder uriBuilder, final NRConnection.Listener listener) {
        if (mSessionId != null) {
            uriBuilder.appendQueryParameter("sid", mSessionId);
            NRConnection.getInstance().connectionWithRequest(uriBuilder.build(), listener);
        } else {
            hello(new NRConnection.Listener() {
                @Override
                public void response(Object responseParam, int status, NRError error) {
                    executeRequest(uriBuilder, listener);
                }

                @Override
                public void log(String tag, String msg) {
                    nrLogger.log(tag, msg);
                }
            });
        }
    }

    private void hello(final NRConnection.Listener listener) {
        final Uri.Builder _uriBuilder = mAccountParams.getUri();
        _uriBuilder.appendQueryParameter("kb", mAccountParams.getKnowledgeBase());
        _uriBuilder.appendEncodedPath("api/widget/v1/hello.js");
        _uriBuilder.appendQueryParameter("nostats", "false");
        _uriBuilder.appendQueryParameter("url", "mobile");
        String context = mAccountParams.getContext();
        if (context != null) {
            _uriBuilder.appendQueryParameter("context", context);
        }
        NRConnection.getInstance().connectionWithRequest(_uriBuilder.build(), new NRConnection.Listener() {
            @Override
            public void response(Object responseParam, int status, NRError error) {
                if (error != null) {
                    Log.e("Nanorep", error.getDescription());
                } else {
                    mSessionId = (String) ((HashMap) responseParam).get("sessionId");
                    mDelay = ((Integer) ((HashMap) responseParam).get("timeout")).longValue() * 500;
                    if(!keepAliveRunning) {
                        keepAliveRunning = true;
                        keepAlive(mDelay);
                    }

                    listener.response(responseParam, status, error);
                }
            }

            @Override
            public void log(String tag, String msg) {
                nrLogger.log(tag, msg);
            }
        });
    }

    public void removeFromSearchCache(String text) {
        if (mCachedSearches.containsKey(text)) {
            mCachedSearches.remove(text);
        }
    }

    public void searchText(final String text, final OnSearchResultsFetchedListener onSearchResultsFetchedListener) {
        if (mCachedSearches != null && mCachedSearches.get(text) != null) {
            onSearchResultsFetchedListener.onSearchResponse(mCachedSearches.get(text), null);
        } else if (text != null && text.length() > 0) {
            String encodedText = null;
            try {
                encodedText = URLEncoder.encode(text, "UTF-8");
            } catch (UnsupportedEncodingException e) {
                e.printStackTrace();
            }
            Uri.Builder uriBuilder = mAccountParams.getUri();
            uriBuilder.appendEncodedPath("api/widget/v1/q.js");
            uriBuilder.appendQueryParameter("text", text);
            executeRequest(uriBuilder, new NRConnection.Listener() {
                @Override
                public void response(Object responseParam, int status, NRError error) {
                    if (error != null) {
                        onSearchResultsFetchedListener.onSearchResponse(null, error);
                    } else if (responseParam != null) {
                        NRSearchResponse response = new NRSearchResponse((HashMap) responseParam);
                        Nanorep.this.getCachedSearches().put(text, response);
                        onSearchResultsFetchedListener.onSearchResponse(response, null);
                    }
                }

                @Override
                public void log(String tag, String msg) {
                    nrLogger.log(tag, msg);
                }
            });
        }
    }

    public void getContextValues(final String context, OnDataResponse<NRContextList> listener) {
        Uri.Builder uriBuilder = mAccountParams.getUri();
        uriBuilder.appendEncodedPath("api/kb/v1/contextValues");
        uriBuilder.appendQueryParameter("contexts", context);
        uriBuilder.appendQueryParameter("account", getAccountParams().getAccount());
        uriBuilder.appendQueryParameter("kb", getAccountParams().getKnowledgeBase());
        uriBuilder.appendQueryParameter("cb", "");
        NRConnection.connection(uriBuilder, listener);
    }

    public void setContext(final String context, final OnDataResponse<DefaultResponse> listener) {
        if (mSessionId == null) {
            if(context == null) {
                return;
            }
            final Uri.Builder uri = mAccountParams.getUri();
            uri.appendQueryParameter("kb", mAccountParams.getKnowledgeBase());
            uri.appendEncodedPath("api/widget/v1/hello.js");
            uri.appendQueryParameter("nostats", "false");
            uri.appendQueryParameter("url", "mobile");
            if (mAccountParams.getContext() != null) {
                uri.appendQueryParameter("context", mAccountParams.getContext());
            }

            NRConnection.connection(uri, new OnDataResponse<HelloResponse>() {
                @Override
                public void onSuccess(HelloResponse response) {
                    mSessionId = response.getSessionId();
                    setSessionParams(context, listener);
                    keepAlive(response.getTimeout()*1000/2);
                }

                @Override
                public void onError(NRConnectionException error) {

                }
            });


        } else {
            setSessionParams(context, listener);
        }
    }

    private void setSessionParams(String context, OnDataResponse<DefaultResponse> listener) {
        if(context == null) {
            clearCache();
            mAccountParams.clearSelectedContext();
        }
        else {
            mAccountParams.setSelectedContext(context);
            context = mAccountParams.getSelectedContext();
        }
        clearCache();
        String initContext = mAccountParams.getContext();
        if(initContext != null) {
            context = initContext + "," + context;
        }
        Uri.Builder uriBuilder = mAccountParams.getUri();
        uriBuilder.appendEncodedPath("api/widget/v1/setParams");
        uriBuilder.appendQueryParameter("context", context == null ? "clearContext:" : context);
        uriBuilder.appendQueryParameter("sid", mSessionId);
        uriBuilder.appendQueryParameter("account", getAccountParams().getAccount());
        NRConnection.connection(uriBuilder, listener);
    }

    public void getChanneling(String articleId, boolean isPositive, final OnDataResponse<List<NRChanneling>> listener) {
        Uri.Builder uriBuilder = mAccountParams.getUri();
        uriBuilder.appendEncodedPath("api/channeling/v1/get");
        uriBuilder.appendQueryParameter("account", getAccountParams().getAccount());
        uriBuilder.appendQueryParameter("kb", getAccountParams().getKnowledgeBase());
        uriBuilder.appendQueryParameter("articleId", articleId);
        uriBuilder.appendQueryParameter("clientState", isPositive ? "positiveFeedback" : "negativeFeedback");

        if (mAccountParams.getCurrentLabelContext() != null) {
            uriBuilder.appendQueryParameter("context", mAccountParams.getCurrentLabelContext());
        }
//        NRConnection.connection(uriBuilder, listener);

        executeRequest(uriBuilder, new NRConnection.Listener() {
            @Override
            public void response(Object responseParam, int status, NRError error) {
                if (error != null) {
                    listener.onError(null);
                } else if (responseParam != null) {
                    List<HashMap<String, Object>> channeling = (List<HashMap<String, Object>>) ((HashMap) responseParam).get("rechanneling");

                    if(channeling == null) {
                        return;
                    }

                    List<NRChanneling> nrChannels = new ArrayList<>();
                    for (HashMap<String, Object> channel : channeling) {
                        nrChannels.add(NRChanneling.channelForParams(channel));
                    }
                    listener.onSuccess(nrChannels);
                }
            }


            @Override
            public void log(String tag, String msg) {
                nrLogger.log(tag, msg);
            }
        });
    }

    public void updateFaqList(String hexLabelId, String nrContext, boolean ignoreLabelContext, NRConnection.Listener listener) {
        AccountParams accountParams = Nanorep.getInstance().getAccountParams();
        final Uri.Builder uri = accountParams.getUri();
        uri.appendEncodedPath("api/kb/v1/faq");
        uri.appendQueryParameter("account", accountParams.getAccount());
        uri.appendQueryParameter("kb", accountParams.getKnowledgeBase());
        if(!accountParams.isContextExclusivity()) {
            uri.appendQueryParameter("allContextsMandatory", "false");
        }
        uri.appendQueryParameter("items", "1000");
        if (hexLabelId != null) {
            uri.appendQueryParameter("labelId", hexLabelId);
        }
//        else if(accountParams.isLabelsMode() == false && accountParams.getLabelId() != 0){
//            uri.appendQueryParameter("labelId", Integer.toHexString(accountParams.getLabelId()));
//            uri.appendQueryParameter("allContextsMandatory", "true");
//        }

        String ctx = null;
        String currentContext = accountParams.getContext();

        if (ignoreLabelContext) {
            ctx = nrContext;
        } else {
            mAccountParams.setCurrentLabelContext(nrContext);
            if (currentContext != null && currentContext.length() > 0) {
                ctx = currentContext + (nrContext != null && nrContext.length() > 0 ? "," + nrContext : "");
            } else if (nrContext != null && nrContext.length() > 0) {
                ctx = nrContext;
            }
        }

        try {
            if (ctx != null) {
                String b64ctx = Base64.encodeToString(ctx.getBytes("US-ASCII"), Base64.DEFAULT);
                if (b64ctx.endsWith("\n")) {
                    b64ctx = b64ctx.substring(0, b64ctx.length() - 1);
                }
                uri.appendQueryParameter("context", b64ctx);
            }
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
        }
        accountParams.setLabelsMode(false);
        NRConnection.getInstance().connectionWithRequest(uri.build(), listener);
    }

    public void suggestionsForText(final String text, final OnSuggestionsFetchedListener onSuggestionsFetchedListener) {
        if (mCachedSuggestions != null && mCachedSuggestions.get(text) != null) {
            onSuggestionsFetchedListener.onSuggestionsFetched(mCachedSuggestions.get(text), null);
        } else if (text != null && text.length() > 0) {
//            String encodedText = null;
//            try {
//                encodedText = URLEncoder.encode(text, "UTF-8");
//            } catch (UnsupportedEncodingException e) {
//                e.printStackTrace();
//            }
            Uri.Builder uriBuilder = mAccountParams.getUri();
            uriBuilder.appendEncodedPath("api/kb/v1/autoComplete");
            uriBuilder.appendQueryParameter("text", text);
            uriBuilder.appendQueryParameter("stemming", "true");
//            String context = mAccountParams.getContext();
//            if (context != null) {
//                uriBuilder.appendQueryParameter("context", context);
//            }
            executeRequest(uriBuilder, new NRConnection.Listener() {
                @Override
                public void response(Object responseParam, int status, NRError error) {
                    if (responseParam != null) {
                        ArrayList<String> answers = (ArrayList) ((HashMap) responseParam).get("a");
                        searchQuery = text;
                        if (answers != null) {
                            ArrayList<Spannable> suggestions = getSpannables(answers);

                            ((HashMap) responseParam).put("a", suggestions);
                            getCachedSuggestions().put(text, new NRSuggestions((HashMap) responseParam));
                            onSuggestionsFetchedListener.onSuggestionsFetched(Nanorep.this.mCachedSuggestions.get(text), null);
                        } else {
                            onSuggestionsFetchedListener.onSuggestionsFetched(null, NRError.error("nanorep", 1002, "No suggestions"));
                        }

                    } else if (error != null) {
                        NRSuggestions storedSuggestions = getCachedSuggestions().get(text);
                        if (storedSuggestions == null) {
                            onSuggestionsFetchedListener.onSuggestionsFetched(null, NRError.error("nanorep", 1003, "No suggestions in cache"));
                        } else {
                            onSuggestionsFetchedListener.onSuggestionsFetched(storedSuggestions, null);
                        }
                    }
                }

                @Override
                public void log(String tag, String msg) {
                    nrLogger.log(tag, msg);
                }
            });
        }
    }

    private ArrayList<Spannable> getSpannables(ArrayList<String> answers) {
        ArrayList<Spannable> suggestions = new ArrayList<>();
        for (String answer : answers) {
            suggestions.add(getSpannable(answer));
        }

        return suggestions;
    }

    public Spannable getSpannable(String text) {
        final SpannableStringBuilder str = new SpannableStringBuilder();
        String []originalWords = searchQuery.split("\\s+");
        String[] pipes = text.split("\\|");
        for (int i = 0; i < pipes.length; i++) {
            String[] words = pipes[i].split(",");

            String word = words[0];
            StyleSpan styleSpan;

            if (word.endsWith("*")) {
                word = word.substring(0,word.length() - 1);
                styleSpan = new StyleSpan(isWordInArray(word,originalWords) ? Typeface.NORMAL : Typeface.BOLD);
            } else {
                styleSpan = new StyleSpan(Typeface.NORMAL);
            }

            String wordSpace = word + " ";

            if (i == pipes.length - 1) { // last
                wordSpace = word;
            }

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                str.append(wordSpace, styleSpan, 0);
            } else {
                int startIndex = str.length();
                str.append(wordSpace);
                str.setSpan(styleSpan, startIndex, str.length(), 0);
            }
        }
        return str;
    }

    private boolean isWordInArray(String needle, String[] originalWords) {
        if(originalWords == null || originalWords.length == 0) {
            return false;
        }

        for (String word : originalWords) {
            if(word.equals(needle)) {
                return true;
            }
        }
        return false;
    }

    public void likeForSearchResult(final NRSearchLikeParams likeParams, final OnLikeSentListener onLikeSentListener) {
        Uri.Builder uriBuilder = mAccountParams.getUri();
        uriBuilder.appendEncodedPath("api/widget/v1/thumb.js");
        for (String key : likeParams.getParams().keySet()) {
            uriBuilder.appendQueryParameter(key, likeParams.getParams().get(key));
        }
        executeRequest(uriBuilder, new NRConnection.Listener() {
            @Override
            public void response(Object responseParam, int status, NRError error) {
                if (error != null) {
                    onLikeSentListener.onLikeSent(false);
                } else if (responseParam != null) {
                    onLikeSentListener.onLikeSent(true);
                }
            }

            @Override
            public void log(String tag, String msg) {
                nrLogger.log(tag, msg);
            }
        });
    }

    public void fetchFAQAnswer(final String answerId, final Integer answerHash, final OnFAQAnswerFetchedListener onFAQAnswerFetchedListener) {
        Uri.Builder uriBuilder = mAccountParams.getUri();
        uriBuilder.appendEncodedPath("api/faq/v1/answer.js");
        uriBuilder.appendQueryParameter("id", answerId);
        uriBuilder.appendQueryParameter("referer", getAccountParams().getReferrer());
        uriBuilder.appendQueryParameter("kb", getAccountParams().getKnowledgeBase());

        // if exist and updated in cache, fetch from cache,
        // else call to server

        // check if we already called fetch answer for this answer id
        ArrayList<OnFAQAnswerFetchedListener> onFAQAnswerFetchedListenerArr = Nanorep.this.getFaqRequestListenersMap().get(answerId);

        if (onFAQAnswerFetchedListenerArr == null) {// array is empty, no requests for this answer id

            onFAQAnswerFetchedListenerArr = new ArrayList<OnFAQAnswerFetchedListener>();

            onFAQAnswerFetchedListenerArr.add(onFAQAnswerFetchedListener);
            Nanorep.this.getFaqRequestListenersMap().put(answerId, onFAQAnswerFetchedListenerArr);

            final ArrayList<OnFAQAnswerFetchedListener> finalOnFAQAnswerFetchedListenerArr = onFAQAnswerFetchedListenerArr;


            executeRequest(uriBuilder, new NRConnection.Listener() {
                @Override
                public void response(Object responseParam, int status, NRError error) {
                    if (responseParam != null) {
                        ((HashMap<String, Object>) responseParam).put("id", answerId);
                    }

                    for (OnFAQAnswerFetchedListener listener : finalOnFAQAnswerFetchedListenerArr) {
                        if (error != null) {
                            listener.onFAQAnswerFetched(null, error);
                        } else if (responseParam != null) {

                            listener.onFAQAnswerFetched(new NRFAQAnswer((HashMap<String, Object>) responseParam), null);
                        }
                    }


                    Nanorep.this.getFaqRequestListenersMap().remove(answerId);
                }

                @Override
                public void log(String tag, String msg) {
                    nrLogger.log(tag, msg);
                }
            });
        }
    }

    public void likeForFAQResult(final NRFAQLikeParams likeParams, final OnLikeSentListener onLikeSentListener) {
        Uri.Builder uriBuilder = mAccountParams.getUri();
        uriBuilder.appendEncodedPath("api/analytics/v1/addFeedback");
        uriBuilder.appendQueryParameter("ignoreValidateCookie", "true");
        if (mAccountParams.getKnowledgeBase() != null) {
            uriBuilder.appendQueryParameter("kb", mAccountParams.getKnowledgeBase());
        }
        for (String key : likeParams.getParams().keySet()) {
            uriBuilder.appendQueryParameter(key, likeParams.getParams().get(key));
        }
        if (mSessionId != null) {
            uriBuilder.appendQueryParameter("sid", mSessionId);
        }
        NRConnection.getInstance().connectionWithRequest(uriBuilder.build(), new NRConnection.Listener() {
            @Override
            public void response(Object responseParam, int status, NRError error) {
                if (error != null) {
                    onLikeSentListener.onLikeSent(false);
                } else if (responseParam instanceof HashMap) {
                    onLikeSentListener.onLikeSent(true);
                }
            }

            @Override
            public void log(String tag, String msg) {
                nrLogger.log(tag, msg);
            }
        });
    }

    public void fetchConfiguration() {
        if (mAccountParams != null) {
            final Uri.Builder uri = mAccountParams.getUri(false);
            uri.appendEncodedPath("widget/scripts/cnf.json");
            if (mAccountParams.getKnowledgeBase() != null) {
                uri.appendQueryParameter("kb", mAccountParams.getKnowledgeBase());
            }
            uri.appendQueryParameter("isFloat", "true");
            uri.appendQueryParameter("context", mAccountParams.getContext());
            uri.appendQueryParameter("Referer", NRUtilities.buildReferer(getAccountParams().getReferrer()));

            NRConnection.getInstance().connectionWithRequest(uri.build(), new NRConnection.Listener() {
                @Override
                public void response(Object responseParam, int status, NRError error) {
                    if(responseParam == null) {
                        widgetListener.onInitializationFailure();
                        return;
                    }
                    Nanorep.this.mCnf = new NRConfiguration((HashMap) responseParam);
                    mAccountParams.setKnowledgeBaseId(mCnf.getKbId());
                    widgetListener.onConfigurationFetched();
                }

                @Override
                public void log(String tag, String msg) {
                    nrLogger.log(tag, msg);
                }
            });
        }
    }

    public NRConfiguration getNRConfiguration() {
        if (mCnf == null)
            mCnf = new NRConfiguration();

        return mCnf;
    }

//    @Override
//    public void setDebugMode(boolean checked) {
//        if (nrLogger == null) {
//            nrLogger = new NRLogger();
//        }
//
//        nrLogger.setDebug(checked);
//    }

    public AccountParams getAccountParams() {
        return mAccountParams;
    }
}
