package com.nanorep.accessibility.voice.engines

import android.content.ContentValues
import android.content.Context
import android.content.Intent
import android.content.pm.PackageManager
import android.os.Bundle
import android.speech.RecognitionListener
import android.speech.RecognizerIntent
import android.speech.SpeechRecognizer
import android.util.Log
import android.view.View
import com.nanorep.accessibility.voice.*
import com.nanorep.accessibility.voice.RecognitionErrorCodes.ERROR_NETWORK
import com.nanorep.accessibility.voice.RecognitionErrorCodes.ERROR_NO_RESULTS
import com.nanorep.accessibility.voice.RecognitionErrorCodes.ERROR_PERMISSIONS
import com.nanorep.accessibility.voice.RecognitionErrorCodes.ERROR_RECOGNIZER_BUSY
import com.nanorep.accessibility.voice.RecognitionErrorCodes.ERROR_RECORDING
import com.nanorep.accessibility.voice.RecognitionErrorCodes.ERROR_TIMEOUT
import com.nanorep.sdkcore.types.NRError
import kotlinx.coroutines.*
import java.util.*

class SpeechToText(context: Context) : VoiceRecognizer(context), SpeechRecognitionProvider {

    private val isSupported:Boolean
    private var speechRecognizer: SpeechRecognizer? = null

    private var speechListener: SpeechedTextListener? = object: SpeechedTextListener {}

    private val recognitionListener: RecognitionListener? = object : RecognitionListener {
        override fun onRmsChanged(rmsdB: Float) {
        }

        override fun onBufferReceived(buffer: ByteArray?) {
            // can display some visual changes while receiving user input
            //Log.d(TAG, "onBufferReceived")
        }

        override fun onPartialResults(partialResults: Bundle?) {
            Log.d(ContentValues.TAG, "onPartialResults")
        }

        override fun onEvent(eventType: Int, params: Bundle?) {
            Log.d(ContentValues.TAG, "onEvent")
        }

        override fun onBeginningOfSpeech() {
            speechListener?.onActive()
        }

        override fun onEndOfSpeech() {
            speechListener?.onIdle()
        }

        override fun onError(error: Int) {
            cancelTimeout()
            val code = when(error) {
                SpeechRecognizer.ERROR_AUDIO -> ERROR_RECORDING
                SpeechRecognizer.ERROR_CLIENT -> ERROR_RECORDING

                SpeechRecognizer.ERROR_INSUFFICIENT_PERMISSIONS -> ERROR_PERMISSIONS

                SpeechRecognizer.ERROR_NETWORK -> ERROR_NETWORK
                SpeechRecognizer.ERROR_NETWORK_TIMEOUT -> ERROR_NETWORK
                SpeechRecognizer.ERROR_SERVER -> ERROR_NETWORK

                SpeechRecognizer.ERROR_NO_MATCH -> ERROR_NO_RESULTS
                SpeechRecognizer.ERROR_SPEECH_TIMEOUT -> ERROR_NO_RESULTS //6

                SpeechRecognizer.ERROR_RECOGNIZER_BUSY -> let {
                    speechRecognizer?.cancel() //8
                    ERROR_RECOGNIZER_BUSY
                }//7

                else -> let{RecognitionErrorCodes.ERROR_GENERAL }
            }
            //toast(context, "failed to get speech data: $message")
            speechListener?.onError(NRError(code, null, code, null))
            //muteBeep(false)
        }

        override fun onResults(results: Bundle?) {
            cancelTimeout()
            Log.d(ContentValues.TAG, "onResults " + results)
            val data: ArrayList<String>? = results?.getStringArrayList(SpeechRecognizer.RESULTS_RECOGNITION)
            if (data == null || data.size == 0) {
                //toast(context, "Got Nothing");
                onError(SpeechRecognizer.ERROR_NO_MATCH)
                return;
            }
            // currently only first result will be used
            val fullText: String = data.get(0)// .joinToString("")

            Log.i(VoiceRecognizerTag, "onReadyForSpeech: ready for recoding, user can start talking")
            speechListener?.onResults(fullText)
            //muteBeep(false)
            //mText.setText("results: " + String.valueOf(data.size()))
        }

        override fun onReadyForSpeech(params: Bundle?) {
            cancelTimeout()
            // toast(context, "start talking")
            Log.i(VoiceRecognizerTag, "onReadyForSpeech: ready for recoding, user can start talking")
            speechListener?.onPrepared()
        }


    }

    private var timeoutJob: Job? = null
    private var listeningStarted = false


    init {
        isSupported = isSpeechToTextAvailable(context)
        if(isSupported) {
            speechRecognizer = SpeechRecognizer.createSpeechRecognizer(context);
            speechRecognizer?.setRecognitionListener(recognitionListener)
        }

    }

    private fun isSpeechToTextAvailable(context: Context) : Boolean {

        return SpeechRecognizer.isRecognitionAvailable(context) && context.getPackageManager().hasSystemFeature(PackageManager.FEATURE_MICROPHONE)
    }

    //  fun bindResultsView(textView: TextView) = apply { this.textView = textView }

    fun bindActionView(view: View) = apply {
        this.actionView = view
        actionView?.setOnClickListener {
            this.start()
        }
    }


    override fun registerListener(speechedTextListener: SpeechedTextListener) {
        apply {
            this.speechListener = speechedTextListener
            if (!isSupported) {
                speechListener?.onError(NRError(VOICE_RECOGNITION_NOT_AVAILABLE, "Speech to text is not available on your device"))
            }
        }
    }

    override fun start(languge: Locale):Unit{
        val recognizeIntent = Intent(RecognizerIntent.ACTION_RECOGNIZE_SPEECH)
        recognizeIntent.putExtra(RecognizerIntent.EXTRA_LANGUAGE_MODEL, RecognizerIntent.LANGUAGE_MODEL_FREE_FORM)
        //  recognizeIntent.putExtra(RecognizerIntent.EXTRA_PROMPT, "please start talking")
        //recognizeIntent.putExtra(RecognizerIntent.EXTRA_SPEECH_INPUT_COMPLETE_SILENCE_LENGTH_MILLIS, 1000L)
        recognizeIntent.putExtra(RecognizerIntent.EXTRA_MAX_RESULTS, 3)
        recognizeIntent.putExtra(RecognizerIntent.EXTRA_LANGUAGE, languge)
        //recognizeIntent.putExtra(RecognizerIntent.EXTRA_SPEECH_INPUT_POSSIBLY_COMPLETE_SILENCE_LENGTH_MILLIS, 1000L)

        speechListener?.onStart()
        speechRecognizer?.cancel()
        speechRecognizer?.startListening(recognizeIntent)  //setListener should be applied by now.

        onListeningStart()
    }

    private fun onListeningStart() {
        listeningStarted = true

        setSpeechSilenceTimeout()
    }

    private fun onListeningEnd() {
        listeningStarted = false
        speechListener?.onStop()
    }

    /**
     * in case timeout error is not passed from the speechListener
     */
    private fun setSpeechSilenceTimeout() {
        cancelTimeout()
        timeoutJob = GlobalScope.launch(Dispatchers.Default){
            delay(6000)
            withContext(Dispatchers.Main) {
                speechListener?.onError(NRError(ERROR_TIMEOUT/*SpeechRecognizer.ERROR_SPEECH_TIMEOUT*/, "recognizer failed to get voice input"))
                speechRecognizer?.cancel()
                onListeningEnd()
            }
        }
    }

    override fun stop() {
        Log.i(VoiceRecognizerTag, "stop: $speechRecognizer")

        cancelTimeout()

        if(listeningStarted) {
            speechRecognizer?.stopListening()
            onListeningEnd()
        }
    }

    /*private fun muteBeep(mute: Boolean){
        val audioManager = context.getSystemService(Context.AUDIO_SERVICE) as AudioManager
        audioManager.setStreamMute(AudioManager.STREAM_MUSIC, mute);
    }*/


    override fun cancel() {
        Log.i(VoiceRecognizerTag, "cancel: $speechRecognizer")

        cancelTimeout()
        speechRecognizer?.cancel()
        onListeningEnd()
    }

    override fun release() {
        Log.i(VoiceRecognizerTag, "release: $speechRecognizer, isSupported = $isSupported")

        cancelTimeout()//cancel()
        speechRecognizer?.setRecognitionListener(null)

        try {
            speechRecognizer?.destroy()
        } catch (e: Exception) {
            e.printStackTrace()
            Log.e("srDestroyException", "speechRecognizer destroy failed")
        }
        onListeningEnd()
    }

    private fun cancelTimeout() {
        if(timeoutJob != null && timeoutJob!!.isActive) {
            timeoutJob?.cancel()
        }
    }

    override fun enable(enable: Boolean) {
        if(!enable) speechRecognizer?.cancel()
    }

}