@file:JvmName("VoiceRecognizerSpace")

package com.nanorep.accessibility.voice

import android.content.Context
import android.os.Build
import android.speech.tts.TextToSpeech
import android.view.View
import android.widget.TextView
import com.nanorep.sdkcore.types.NRError
import java.util.*


/**
 * Created by tehila on 27/11/2017.
 */

const val AUDIO_RECORD_PERMISSION_CODE:Int = 309
const val VOICE_RECOGNITION_NOT_AVAILABLE = "VOICE_RECOGNITION_NOT_AVAILABLE"
const val TEXT_TO_VOICE_NOT_AVAILABLE = "TEXT_TO_VOICE_NOT_AVAILABLE"
const val VoiceRecognizerTag = "VoiceRecognition"

object RecognitionErrorCodes {
    const val ERROR_RECORDING = "VOICE_ERROR_RECORDING"
    const val ERROR_PERMISSIONS = "VOICE_ERROR_PERMISSIONS"
    const val ERROR_NETWORK = "VOICE_ERROR_NETWORK"
    const val ERROR_NO_RESULTS = "VOICE_ERROR_NO_RESULTS"
    const val ERROR_RECOGNIZER_BUSY = "VOICE_ERROR_RECOGNIZER_BUSY"
    const val ERROR_GENERAL = "VOICE_ERROR_GENERAL"
    const val ERROR_TIMEOUT = "VOICE_ERROR_TIMEOUT"
}


abstract class VoiceRecognizer {

    protected var textView: TextView? = null
        protected set
    protected var actionView: View? = null
        protected set

    protected constructor(context: Context) {
        this.context = context
    }

    protected var context: Context

    fun bindOutputView(textView: TextView) = apply { this.textView = textView }

}

class TextToSpeech(context: Context, var textedListener: TextedSpeechListener? = null) : VoiceRecognizer(context) {

    val innerTextToSpeech : android.speech.tts.TextToSpeech
    private var language: Locale
    private var supported = true

    private val onInitListener = TextToSpeech.OnInitListener() { status ->
        var error = when (status) {

            TextToSpeech.SUCCESS -> null

            TextToSpeech.LANG_NOT_SUPPORTED -> "language not supported"

            TextToSpeech.LANG_MISSING_DATA -> "language not available"

            else -> {
                "some error occurred"
            }

        }

        if (error == null) {
            // we can set language to the TextToSpeech object
            ttsReady()
        } else {
            textedListener?.onTTSError(NRError(TEXT_TO_VOICE_NOT_AVAILABLE, error))
        }
    }

    init {
        innerTextToSpeech = android.speech.tts.TextToSpeech(context, onInitListener)
        language = Locale.getDefault()
    }

    private fun assertServiceAvailable(context: Context) {

        supported = innerTextToSpeech.engines?.isNotEmpty()?:false && let {

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                innerTextToSpeech.availableLanguages?.isNotEmpty() ?: false
            } else {
                innerTextToSpeech.isLanguageAvailable(language) != TextToSpeech.LANG_NOT_SUPPORTED
            }
        }


        /*val ttsIntent = Intent()
        ttsIntent.action = TextToSpeech.Engine.ACTION_CHECK_TTS_DATA

        val pm = context.packageManager

        val list = pm.queryIntentActivities(ttsIntent, PackageManager.GET_META_DATA)*/
    }

    private var ready: Boolean = false

    private fun ttsReady() {
        if(ready) return

        this.ready = true
        innerTextToSpeech.setLanguage(language)

        assertServiceAvailable(context)

        if(supported) {
            textedListener?.onTTSReady(this)
        }
    }


    /*
            fun bindResultsView(textView: TextView) = apply { this.resultsView = textView }

            fun bindActionView(view: View) = apply { this.actionView = view }
    */

    fun language(languge: Locale) = apply {
        this.language = languge
    }

    /**
     * change listener given on constructor
     */
    fun setListener(listener: TextedSpeechListener = object : TextedSpeechListener {}) = apply {
        this.textedListener = listener
        if (!supported) {
            textedListener?.onTTSError(NRError(TEXT_TO_VOICE_NOT_AVAILABLE, "language not supported"))
        }
    }

    fun speak(text : String):Unit {
        if(!ready) return

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            innerTextToSpeech.speak(text, TextToSpeech.QUEUE_FLUSH, null, null)
        } else {
            innerTextToSpeech.speak(text, TextToSpeech.QUEUE_FLUSH, null)
        }
    }

}


interface TextedSpeechListener{
    fun onTTSReady(textToSpeech: com.nanorep.accessibility.voice.TextToSpeech){}
    fun onTTSError(error: NRError){}
}

interface SpeechRecognitionProvider {
    fun registerListener(speechedTextListener: SpeechedTextListener = object: SpeechedTextListener {}){}
    fun enable(enable:Boolean){}
    fun start(languge: Locale = Locale.getDefault()){}
    fun stop(){}
    fun cancel(){}
    fun release(){}
}

interface SpeechedTextListener {
    fun onReady(){} // called once when speech engine was set
    fun onStart(){} // when voice listening requested
    fun onPrepared(){} // engine started listening
    fun onActive(){} // recorder recognized voice
    fun onIdle(){} // recorder recognized voice end
    fun onResults(text:String){} // voice recognized results
    fun onError(error: NRError) {}
    fun onStop() {}
}

