package com.najva.sdk.push_notification;

import android.content.Context;

import androidx.annotation.NonNull;

import com.google.android.gms.tasks.OnSuccessListener;
import com.google.firebase.FirebaseApp;
import com.google.firebase.FirebaseOptions;
import com.google.firebase.iid.FirebaseInstanceId;
import com.google.firebase.iid.InstanceIdResult;
import com.najva.sdk.ClientMetaData;
import com.najva.sdk.Najva;
import com.najva.sdk.NajvaClient;
import com.najva.sdk.campaign.CampaignController;
import com.najva.sdk.campaign.CampaignControllerImpl;
import com.najva.sdk.core.utils.FileManager;
import com.najva.sdk.core.utils.Logger;
import com.najva.sdk.core.utils.Parameter;
import com.najva.sdk.core.utils.StaticFields;
import com.najva.sdk.location.LocationControllerImpl;

/**
 * created for initialize FirebaseApp
 */

public class PushNotificationControllerImpl extends Thread implements PushNotificationController {

    private static final String TAG = "PushNotification";

    private final Context context;
    private final String apiKey;
    private final int websiteId;
    private String najvaToken;

    public PushNotificationControllerImpl(@NonNull Context context, @NonNull String apiKey,
                                          int websiteId, String najvaToken) {
        this.context = context;
        this.apiKey = apiKey;
        this.websiteId = websiteId;
        this.najvaToken = najvaToken;
    }

    @Override
    public void load() {

        FirebaseOptions options = new FirebaseOptions.Builder()
                .setApplicationId(Parameter.FIREBASE_APPLICATION_ID.getValue()) // Required for Analytics.
                .setApiKey(Parameter.FIREBASE_API_KEY.getValue()) // Required for Auth.
                .setDatabaseUrl(Parameter.FIREBASE_DATABASE_URL.getValue()) // Required for RTDB.
                .setProjectId(Parameter.FIREBASE_PROJECT_ID.getValue())
                .build();

        if (!NajvaClient.configuration.isFirebaseEnabled()) {
            try {
                FirebaseApp.getInstance();
                Logger.i("FirebaseInitialization", "default firebase app found, registering najva in other scope");
                FirebaseApp.initializeApp(context, options, StaticFields.NAJVA);
            } catch (Exception e) {
                Logger.i("FirebaseInitialization", "no default firebase app found, initializing najva as default");
                FirebaseApp.initializeApp(context, options);
                NajvaClient.najvaDefault = true;
            }
        }

        registerIfNeeded(context);
    }

    private void registerIfNeeded(final Context context) {
        //checking if firebase token is available
        if (!NajvaClient.configuration.isFirebaseEnabled() && !NajvaClient.najvaDefault) {
            FirebaseInstanceId
                    .getInstance(FirebaseApp.getInstance(StaticFields.NAJVA))
                    .getInstanceId()
                    .addOnSuccessListener(new OnSuccessListener<InstanceIdResult>() {
                        @Override
                        public void onSuccess(InstanceIdResult instanceIdResult) {
                            register(context, instanceIdResult.getToken());
                        }
                    });
        } else {
            FirebaseInstanceId
                    .getInstance()
                    .getInstanceId()
                    .addOnSuccessListener(new OnSuccessListener<InstanceIdResult>() {
                        @Override
                        public void onSuccess(InstanceIdResult instanceIdResult) {
                            register(context, instanceIdResult.getToken());
                        }
                    });
        }
    }

    private void register(Context context, String firebaseToken) {
        String lastToken = FileManager.readFromFile(context, Parameter.FIREBASE_TOKEN.getValue());
        if (!isRegistrationNeeded(firebaseToken, najvaToken, lastToken)) return;

        // firebase token is available and najva token is not so we must register device

        Logger.i(TAG, "Registering firebase token to najva");

        ClientMetaData.setApiKey(context, this.apiKey);
        ClientMetaData.setToken(firebaseToken);
        ClientMetaData.setWebsiteId(websiteId);
        CampaignController campaignController = new CampaignControllerImpl(context,
                new LocationControllerImpl(context),
                ClientMetaData.getParams());
        campaignController.loadAsync();
    }

    private boolean isRegistrationNeeded(String firebaseToken, String najvaToken, String lastFirebaseToken) {
        if (firebaseToken == null) return false;
        if (!firebaseToken.equals(lastFirebaseToken)) return true;
        return najvaToken == null && !firebaseToken.isEmpty();
    }


    private FirebaseApp getFireBaseInstance() {
        try {
            return FirebaseApp.getInstance();
        } catch (Exception ignored) {
            return FirebaseApp.getInstance(StaticFields.NAJVA);
        }
    }

    @Override
    public void loadAsync() {
        start();
    }

    @Override
    public void run() {
        load();
    }
}
