package com.najva.sdk.core.utils;

import androidx.annotation.NonNull;

import com.android.volley.AuthFailureError;
import com.android.volley.DefaultRetryPolicy;
import com.android.volley.NetworkResponse;
import com.android.volley.ParseError;
import com.android.volley.Response;
import com.android.volley.VolleyError;
import com.android.volley.toolbox.HttpHeaderParser;
import com.android.volley.toolbox.JsonObjectRequest;
import com.android.volley.toolbox.RequestFuture;
import com.najva.sdk.BuildConfig;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.UnsupportedEncodingException;
import java.util.HashMap;
import java.util.Map;

public class MetaJsonRequest extends JsonObjectRequest {
    private static final String TAG = "MetaJsonRequest";
    private static final String HEADER = "headers";

    private static final String CONTENT_TYPE = "Content-Type";
    private CookieManager cookieManager;

    private Map<String, String> params = new HashMap<>();
    private Map<String, String> headers = new HashMap<>();
    private RequestFuture<JSONObject> future;
    private Response.Listener<JSONObject> responseListener;

    private MetaJsonRequest(int method, String url, JSONObject jsonRequest, Response.Listener
            <JSONObject> listener, Response.ErrorListener errorListener) {
        super(method, url, jsonRequest, listener, errorListener);
    }

    private MetaJsonRequest(String url, JSONObject jsonRequest, Response.Listener<JSONObject>
            listener, Response.ErrorListener errorListener) {
        super(url, jsonRequest, listener, errorListener);
    }

    public void setCookieManager(CookieManager cookieManager) {
        this.cookieManager = cookieManager;
    }

    @Override
    protected Response<JSONObject> parseNetworkResponse(NetworkResponse response) {
        try {
            Map<String, String> headers = response.headers;
            String cookies = null;
            try {
                cookies = headers.get("Set-Cookie");
            } catch (Exception ignored) {
            }

            if (cookieManager != null && cookies != null) {
                cookieManager.saveCookie(cookies);
            }

            String jsonString = new String(response.data,
                    HttpHeaderParser.parseCharset(response.headers, PROTOCOL_CHARSET));
            Logger.i("MetaJsonRequest", jsonString);
            JSONObject jsonResponse = new JSONObject(jsonString);
            jsonResponse.put(HEADER, new JSONObject(response.headers));
            return Response.success(jsonResponse,
                    HttpHeaderParser.parseCacheHeaders(response));
        } catch (UnsupportedEncodingException e) {
            return Response.error(new ParseError(e));
        } catch (JSONException je) {
            return Response.error(new ParseError(je));
        }
    }

    @Override
    public Map<String, String> getHeaders() throws AuthFailureError {
        if (headers == null) {
            headers = new HashMap<>();
        }

        if (cookieManager == null)
            return headers;

        headers.put(StaticFields.COOKIE, cookieManager.getCookies());
        headers.put(CONTENT_TYPE, "application/json");
        headers.put(StaticFields.NAJVA_SDK_VERSION, BuildConfig.VERSION_NAME);


        return headers;
    }

    private void addHeader(String key, String value) {
        headers.put(key, value);
    }

    private void addPostParams(String key, String value) {
        params.put(key, value);
    }

    private void setFuture(RequestFuture<JSONObject> future) {
        this.future = future;
    }

    public RequestFuture<JSONObject> getFuture() {
        return future;
    }

    private void setOnResponseListener(Response.Listener<JSONObject> responseListener) {
        this.responseListener = responseListener;
    }

    public void callOnResponse(JSONObject jsonObject) {
        this.responseListener.onResponse(jsonObject);
    }

    public static class Builder implements IBuilder<MetaJsonRequest> {
        private int method = Method.GET;
        private String url;
        private Response.Listener<JSONObject> responseListener = new Response.Listener<JSONObject>() {

            @Override
            public void onResponse(JSONObject response) {
                Logger.i(TAG, "no response listener.");
            }
        };
        private Response.ErrorListener errorListener = new Response.ErrorListener() {
            @Override
            public void onErrorResponse(VolleyError error) {
                Logger.i(TAG, "no error listener.");
            }
        };
        private Map<String, Object> params = new HashMap<>();
        private Map<String, String> headers = new HashMap<>();
        private CookieManager manager;
        private boolean async;

        public Builder setMethod(int method) {
            this.method = method;
            return this;
        }

        public Builder setParams(@NonNull Map<String, Object> params) {
            this.params.putAll(params);
            return this;
        }

        public Builder setUrl(@NonNull String url) {
            this.url = url;
            return this;
        }

        public Builder setResponseListener(@NonNull Response.Listener<JSONObject> listener) {
            this.responseListener = listener;
            return this;
        }

        public Builder setErrorListener(@NonNull Response.ErrorListener listener) {
            this.errorListener = listener;
            return this;
        }

        public Builder addHeader(@NonNull String key, @NonNull String value) {
            headers.put(key, value);
            return this;
        }

        public Builder addParam(@NonNull String key, @NonNull String value) {
            params.put(key, value);
            return this;
        }

        public Builder setCookieManager(@NonNull CookieManager manager) {
            this.manager = manager;
            return this;
        }

        public Builder setAsync(boolean async) {
            this.async = async;
            return this;
        }

        @NonNull
        public MetaJsonRequest build() {
            if (async) {
                return buildAsyncRequest();
            }
            RequestFuture<JSONObject> future = RequestFuture.newFuture();
            MetaJsonRequest request = new MetaJsonRequest(method, url, getJsonParams(), future, errorListener);
            request.setRetryPolicy(new DefaultRetryPolicy(60000,0,0));
            request.setFuture(future);
            request.setOnResponseListener(responseListener);
            for (String key : headers.keySet()) {
                request.addHeader(key, headers.get(key));
            }
//            for (String key : params.keySet()) {
//                request.addPostParams(key, params.get(key));
//            }
            if (manager == null) {
                Logger.d(TAG, "CookieManager is null");
            }
            request.setCookieManager(manager);
            return request;
        }

        @NonNull
        private JSONObject getJsonParams() {
            JSONObject jsonObject = new JSONObject();
            for (String key : params.keySet()) {
                try {
                    jsonObject.put(key, params.get(key));
                } catch (JSONException e) {
                    //e.printStackTrace();
                }
            }
            return jsonObject;
        }

        @NonNull
        private MetaJsonRequest buildAsyncRequest() {
            MetaJsonRequest request = new MetaJsonRequest(method, url, getJsonParams(), responseListener, errorListener);
            for (String key : headers.keySet()) {
                request.addHeader(key, headers.get(key));
            }
//            for (String key : params.keySet()) {
//                request.addPostParams(key, params.get(key));
//            }
            if (manager == null) {
                Logger.d(TAG, "CookieManager is null");
            }
            request.setCookieManager(manager);
            return request;
        }

        public Builder setHeaders(@NonNull Map<String, String> headers) {
            this.headers.putAll(headers);
            return this;
        }
    }


}