package com.najva.sdk;

import android.app.Activity;
import android.app.Application;
import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.content.Context;
import android.content.IntentFilter;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.graphics.Color;
import android.media.RingtoneManager;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;
import androidx.work.WorkInfo;
import androidx.work.WorkManager;

import com.najva.sdk.core.ActivityStack;
import com.najva.sdk.core.SessionManager;
import com.najva.sdk.core.ad.AdIdFetchTask;
import com.najva.sdk.core.ad.AdvertisingIdClient;
import com.najva.sdk.core.utils.FileManager;
import com.najva.sdk.core.utils.Logger;
import com.najva.sdk.core.utils.Parameter;
import com.najva.sdk.core.utils.StaticFields;
import com.najva.sdk.location.LocationController;
import com.najva.sdk.location.LocationControllerImpl;
import com.najva.sdk.login.LoginController;
import com.najva.sdk.login.LoginControllerImpl;
import com.najva.sdk.push_notification.PushNotificationController;
import com.najva.sdk.push_notification.PushNotificationControllerImpl;
import com.najva.sdk.runnables.ClickReceiverHandler;
import com.najva.sdk.runnables.NajvaSubscriberTokenHandler;
import com.najva.sdk.runnables.NotificationReceiverHandler;
import com.najva.sdk.utils.MetaDataUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.ExecutionException;


public class NajvaClient implements Application.ActivityLifecycleCallbacks {

    public static AdvertisingIdClient.AdInfo adInfo;

    public static NajvaConfiguration configuration;

    public static final String TAG = "NajvaClient";

    private static NajvaClient instance;

    public static NajvaClient getInstance() {
        if (instance == null) {
            throw new RuntimeException("You must call NajvaClient.getInstance(Context appContext, NajvaConfiguration configuration); first.");
        }
        return instance;
    }

    @NonNull
    public static NajvaClient getInstance(@NonNull Context context, @Nullable NajvaConfiguration configuration) {
        if (instance == null) {
            if (configuration == null) {
                instance = new NajvaClient(context);
            } else {
                instance = new NajvaClient(context, configuration);
            }
        }
        return instance;
    }

    public static NajvaClient getInstance(@NonNull Context context,
                                          @Nullable NajvaConfiguration configuration,
                                          @NonNull String apiKey,
                                          int websiteId) {
        if (instance == null) {
            if (configuration == null) {
                instance = new NajvaClient(context, apiKey, websiteId);
            } else {
                instance = new NajvaClient(context, configuration, apiKey, websiteId);
            }
        }
        return instance;
    }

    private String apiKey;
    private int websiteId;

    public static boolean najvaDefault = false;

    private final Context appContext;

    private final List<Application.ActivityLifecycleCallbacks> callbacks = new ArrayList<>();

    private NajvaClientReceiver receiver;

    private ActivityStack activityStack;

    private NajvaClient(Context appContext) {
        this(appContext, new NajvaConfiguration());
    }

    private NajvaClient(Context context, NajvaConfiguration configuration) {
        this(context, configuration, null, 0);
    }

    private NajvaClient(Context context, String apiKey, int websiteId) {
        this(context, new NajvaConfiguration(), apiKey, websiteId);
    }

    private NajvaClient(Context context, NajvaConfiguration configuration, String apiKey, int websiteId) {
        appContext = context;
        NajvaClient.configuration = configuration;

        initialize(apiKey, websiteId);

        new AdIdFetchTask(appContext).execute();
    }

    @Nullable
    public String getSubscribedToken() {
        String token = FileManager.readFromFile(appContext, Parameter.TOKEN_FILE_NAME.getValue());
        if (token == null || token.isEmpty())
            return null;
        else {
            return token;
        }
    }

    private void initialize(String apiKey, int websiteId) {
        initializeLogger();

        initializeManifestMetadata(apiKey, websiteId);

        initializeLocation();

        initializeFirebase();

        initializeBroadCast();
        activityStack = new ActivityStack();
        callbacks.add(activityStack);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            createNotificationChannels();
        }

        Logger.i(TAG, "Najva Initialization complete with config: " + configuration.toString());

        try {
            List<WorkInfo> infos = WorkManager.getInstance(appContext)
                    .getWorkInfosByTag(com.najva.sdk.core.utils.WorkManager.TAG).get();

            for (WorkInfo info : infos){
                Logger.d("WorkState", info.getState().name());
            }
        } catch (ExecutionException e) {
            e.printStackTrace();
        } catch (InterruptedException e) {
            e.printStackTrace();
        }


        WorkManager.getInstance(appContext).cancelAllWorkByTag(com.najva.sdk.core.utils.WorkManager.TAG);

    }

    public void setLogEnabled(boolean logEnabled){
        Logger.logEnabled = logEnabled;
    }

    @RequiresApi(api = Build.VERSION_CODES.O)
    private void createNotificationChannels() {
        NotificationManager mNotificationManager = (NotificationManager) appContext.getSystemService(Context.NOTIFICATION_SERVICE);

        assert mNotificationManager != null;

        NotificationChannel lowChannel = new NotificationChannel(StaticFields.CHANNEL_ID_LOW_PRIORITY,
                StaticFields.CHANNEL_DESCRIPTION_LOW, NotificationManager.IMPORTANCE_DEFAULT);

        lowChannel.setShowBadge(true);
        lowChannel.enableLights(true);
        lowChannel.enableVibration(true);
        lowChannel.setSound(RingtoneManager.getDefaultUri(RingtoneManager.TYPE_NOTIFICATION), null);
        lowChannel.setLightColor(Color.CYAN);

        mNotificationManager.createNotificationChannel(lowChannel);

        NotificationChannel highChannel = new NotificationChannel(StaticFields.CHANNEL_ID_HIGH_PRIORITY,
                StaticFields.CHANNEL_DESCRIPTION_HIGH, NotificationManager.IMPORTANCE_HIGH);

        highChannel.setShowBadge(true);
        highChannel.enableLights(true);
        highChannel.enableVibration(true);
        highChannel.setSound(RingtoneManager.getDefaultUri(RingtoneManager.TYPE_NOTIFICATION), null);
        highChannel.setLightColor(Color.CYAN);

        mNotificationManager.createNotificationChannel(highChannel);
    }

    private void initializeLogger() {
        Uri loggerFileUri = Uri.fromFile(appContext.getFilesDir()).buildUpon().appendPath(Parameter.LOGGER_FILE.getValue()).build();
        Logger.setLoggerFilePath(loggerFileUri.getPath());
        Uri indexFileUri = Uri.fromFile(appContext.getFilesDir()).buildUpon().appendPath(Parameter.INDEX_FILE.getValue()).build();
        Logger.setIndexFilePath(indexFileUri.getPath());
        Logger.i(TAG, "Logger initialized");
    }

    private void initializeBroadCast() {
        HashMap<String, NajvaRunnable> actions = new HashMap<>();
        actions.put(StaticFields.EVENT_TYPE_TOKEN, new NajvaSubscriberTokenHandler());
        actions.put(StaticFields.EVENT_NOTIFICATION_RECEIVER, new NotificationReceiverHandler());
        actions.put(StaticFields.EVENT_CLICK, new ClickReceiverHandler());
        receiver = new NajvaClientReceiver(actions);
        appContext.registerReceiver(receiver, new IntentFilter(NajvaClientReceiver.ACTION));
    }

    private void initializeLogin() {
        LoginController loginController = new LoginControllerImpl(appContext, apiKey, websiteId);
        loginController.loadAsync();
    }

    private void initializeManifestMetadata(String apiKey, int websiteId) {
        if (apiKey == null) {
            this.apiKey = getApiKey(appContext);
        } else {
            this.apiKey = apiKey;
        }

        if (websiteId == 0) {
            this.websiteId = getWebsiteId(appContext);
        } else {
            this.websiteId = websiteId;
        }

        ClientMetaData.setWebsiteId(this.websiteId);
        ClientMetaData.setApiKey(appContext, this.apiKey);

        Logger.i(TAG, "Manifest metadata has been read");
        Logger.i(TAG, "ApiKey: " + this.apiKey);
        Logger.i(TAG, "WebsiteId: " + this.websiteId);
    }

    private void initializeFirebase() {
        if (configuration.isPushNotificationEnabled()) {
            PushNotificationController pushNotificationController = new PushNotificationControllerImpl(appContext,
                    apiKey, websiteId, getSubscribedToken());
            pushNotificationController.loadAsync();

        }
    }

    private void initializeLocation() {
        if (configuration.isLocationEnabled()) {
            LocationController locationController = new LocationControllerImpl(appContext);
            locationController.loadAsync();
            Logger.i(TAG, "Location initialized");
        }
    }

    @Override
    public void onActivityCreated(Activity activity, Bundle savedInstanceState) {
        SessionManager.getInstance().onStart();
        if (activityStack.empty()) {
            initializeLogin();
        }
        for (Application.ActivityLifecycleCallbacks callback : callbacks) {
            callback.onActivityCreated(activity, savedInstanceState);
        }
    }


    @Override
    public void onActivityStarted(Activity activity) {
        for (Application.ActivityLifecycleCallbacks callback : callbacks) {
            callback.onActivityStarted(activity);
        }
    }

    @Override
    public void onActivityResumed(Activity activity) {
        for (Application.ActivityLifecycleCallbacks callback : callbacks) {
            callback.onActivityResumed(activity);
        }
    }

    @Override
    public void onActivityPaused(Activity activity) {
        for (Application.ActivityLifecycleCallbacks callback : callbacks) {
            callback.onActivityPaused(activity);
        }
    }

    @Override
    public void onActivityStopped(Activity activity) {
        for (Application.ActivityLifecycleCallbacks callback : callbacks) {
            callback.onActivityStopped(activity);
        }
    }

    @Override
    public void onActivitySaveInstanceState(Activity activity, Bundle outState) {
        for (Application.ActivityLifecycleCallbacks callback : callbacks) {
            callback.onActivitySaveInstanceState(activity, outState);
        }
    }

    @Override
    public void onActivityDestroyed(Activity activity) {
        if (activityStack.isEmpty()) {
            SessionManager.getInstance().onStop();
        }
        for (Application.ActivityLifecycleCallbacks callback : callbacks) {
            callback.onActivityDestroyed(activity);
        }
    }

    public void onAppTerminated() {
        appContext.unregisterReceiver(receiver);
    }

    private int getWebsiteId(@NonNull Context context) {
        return MetaDataUtils.getInstance(getApplicationInfo(context)).getWebsiteId();
    }

    private String getApiKey(@NonNull Context context) {
        return MetaDataUtils.getInstance(getApplicationInfo(context)).getApiKey();
    }

    private ApplicationInfo getApplicationInfo(@NonNull Context context) {
        try {
            return context.getPackageManager().getApplicationInfo(context.getPackageName(), PackageManager.GET_META_DATA);
        } catch (PackageManager.NameNotFoundException e) {
            e.printStackTrace();
        }
        return null;
    }

}
