package com.najva.sdk.core.utils;


import android.content.Context;
import android.net.Uri;

import androidx.annotation.NonNull;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.RandomAccessFile;

public class FileManager {

    /**
     * creates a new file under the files directory of application
     *
     * @param context is context to access files directoy
     * @param name    is the name of the file
     * @return null if permission to read and write to files directory has not granted
     * else will return File pointing to the a file with name of name and directory of filesDir.
     */
    private static File getFile(@NonNull Context context, @NonNull String name, boolean createFile) {
        Uri uri = Uri.fromFile(context.getFilesDir()).buildUpon().appendPath(name).build();
        File file = new File(uri.getPath());
        try {
            if (!file.exists() && createFile) {
                file.getParentFile().mkdirs();
                file.createNewFile();
            }
            return file;
        } catch (IOException e) {
            return null;
        }
    }
    private static File getFile(@NonNull String path, boolean createFile) {
        File file = new File(path);
        try {
            if (!file.exists() && createFile) {
                file.getParentFile().mkdirs();
                file.createNewFile();
            }
            return file;
        } catch (IOException e) {
            return null;
        }
    }

    /**
     * creates a new file under the cache directory of application
     *
     * @param context is context to access files directoy
     * @param name    is the name of the file
     * @return null if permission to read and write to files directory has not granted
     * else will return File pointing to the a file with name of name and directory of cacheDir.
     */
    private static File getCache(@NonNull Context context, @NonNull String name, boolean createFile) {
        Uri uri = Uri.fromFile(context.getCacheDir()).buildUpon().appendPath(name).build();
        File file = new File(uri.getPath());
        try {
            if (!file.exists() && createFile) {
                file.getParentFile().mkdirs();
                file.createNewFile();
            }
            return file;
        } catch (IOException e) {
            return null;
        }
    }

    public static String readFromFile(@NonNull Context context, @NonNull String fileName) {
        try {
            FileInputStream fis = new FileInputStream(getFile(context, fileName, false));
            byte[] bytes = new byte[fis.available()];
            fis.read(bytes);
            fis.close();
            return new String(bytes);
        } catch (FileNotFoundException e) {
            //error read from file : file not found
            return null;
        } catch (IOException e) {
            //error read from file : permission denied
            return null;
        }
    }
    public static String readFromFile(@NonNull String filePath) {
        try {
            FileInputStream fis = new FileInputStream(getFile(filePath, false));
            byte[] bytes = new byte[fis.available()];
            fis.read(bytes);
            fis.close();
            String content = new String(bytes);
            if (content.isEmpty()) return null;
            return content;
        } catch (FileNotFoundException e) {
            //error read from file : file not found
            return null;
        } catch (IOException e) {
            //error read from file : permission denied
            return null;
        }
    }

    public static void writeToFile(@NonNull Context context, @NonNull String fileName, @NonNull String content) {
        try {
            FileOutputStream fos = new FileOutputStream(getFile(context, fileName, true));
            fos.write(content.getBytes());
            fos.flush();
            fos.close();
        } catch (FileNotFoundException e) {
            // error write to file : file not found
            File file = getFile(context, fileName, true);
            try {
                if (file.getParentFile().mkdirs() && file.createNewFile()) {
                    writeToFile(context, fileName, content);
                }
            } catch (IOException e1) {
                // error create new file : permission denied
            }
        } catch (IOException e) {
            // error write to file : permission denied
        }
    }
    public static void writeToFile(@NonNull String filePath, @NonNull String content) {
        try {
            FileOutputStream fos = new FileOutputStream(getFile(filePath, true));
            fos.write(content.getBytes());
            fos.flush();
            fos.close();
        } catch (FileNotFoundException e) {
            // error write to file : file not found
            File file = getFile(filePath, true);
            try {
                if (file.getParentFile().mkdirs() && file.createNewFile()) {
                    writeToFile(filePath, content);
                }
            } catch (IOException e1) {
                // error create new file : permission denied
            }
        } catch (IOException e) {
            // error write to file : permission denied
        }
    }

    public static void appendToRandomAccessFile(@NonNull String filePath, @NonNull String content, int index, int maxLineLength) {
        try {
            RandomAccessFile randomAccessFile = new RandomAccessFile(getFile(filePath, false), "rw");
            randomAccessFile.seek(index * maxLineLength);
            randomAccessFile.write(content.getBytes());
            randomAccessFile.close();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    public static String readFromCache(@NonNull Context context, @NonNull String fileName) {
        try {
            FileInputStream fis = new FileInputStream(getCache(context, fileName, false));
            byte[] bytes = new byte[fis.available()];
            fis.read(bytes);
            fis.close();
            return new String(bytes);
        } catch (FileNotFoundException e) {
            // error read from cache : file not found
            return null;
        } catch (IOException e) {
            // error read from cache : permission denied
            return null;
        }
    }

    public static void writeToCache(@NonNull Context context, @NonNull String filename, @NonNull String content) {
        try {
            FileOutputStream fos = new FileOutputStream(getCache(context, filename, true));
            fos.write(content.getBytes());
            fos.flush();
            fos.close();
        } catch (FileNotFoundException e) {
            // error write to cache : file not found
            File file = getCache(context, filename, true);
            try {
                if (file.getParentFile().mkdirs() && file.createNewFile()) {
                    writeToCache(context, filename, content);
                }
            } catch (IOException e1) {
                // error write to cache : permission denied
            }
        } catch (IOException e) {
            // error write to cache : permission denied
        }
    }

    /**
     * deleting a file from files directory of application
     *
     * @param context  to access files directory
     * @param filename is the name of the file under the files directory
     */
    public static void deleteCache(@NonNull Context context, @NonNull String filename) {
        getCache(context, filename, true).delete();
    }

    /**
     * deleting a file from cache directory of application
     *
     * @param context  to access cache directory
     * @param filename is the name of the file under the cache directory
     */
    public static void deleteFile(@NonNull Context context, @NonNull String filename) {
        getFile(context, filename, true).delete();
    }

    public static void appendToFile(Context context, String fileName, String content) {
        File file = getFile(context, fileName, false);
        try {
            FileOutputStream fis = new FileOutputStream(file, true);
            fis.write(content.getBytes());
            fis.flush();
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }
}
