package com.najva.sdk.push_notification.service;

import android.app.IntentService;
import android.app.NotificationManager;
import android.content.Context;
import android.content.Intent;
import android.net.Uri;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;


import androidx.work.OneTimeWorkRequest;
import androidx.work.WorkManager;

import com.android.volley.Request;
import com.najva.sdk.core.utils.ApiInfo;
import com.najva.sdk.core.utils.CookieManager;
import com.najva.sdk.core.utils.Logger;
import com.najva.sdk.core.utils.StaticFields;
import com.najva.sdk.core.works.FormRequestWorker;
import com.najva.sdk.runnables.ClickReceiverHandler;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.HashMap;

import static com.najva.sdk.core.utils.StaticFields.UUID;

public class NajvaNotificationIntentService extends IntentService {

    private static final String CLICK_ACTION = "onclick_action";
    private static final String FINAL_ADDRESS = "final_address";
    private static final String NOTIFICATION_ID = "notification_id";
    private static final String COMPLETE_URL = "complete_url";
    private static final String NOTIFICATION_CONTENT = "notification_content";

    public static final String OPEN_LINK = "open-link";
    public static final String OPEN_APP = "open-app";
    public static final String OPEN_ACTIVITY = "open-activity";
    public static final String OPEN_CALL = "open-call";
    public static final String OPEN_SMS = "open-sms";
    public static final String OPEN_TELEGRAM_CHANNEL = "open-telegram-channel";
    public static final String JOIN_TELEGRAM_CHANNEL = "join-telegram-channel";

    public static Intent newIntent(@NonNull Context context, @NonNull String notificationAction,
                                   int notificationId,
                                   String finalAddress,
                                   String completeUrl,
                                   String notificationContent,
                                   String uuid,
                                   String json) {

        Intent intent = new Intent(context, NajvaNotificationIntentService.class);
        intent.putExtra(UUID, uuid);
        intent.putExtra(CLICK_ACTION, notificationAction);
        intent.putExtra(FINAL_ADDRESS, finalAddress);
        intent.putExtra(NOTIFICATION_ID, notificationId);
        intent.putExtra(NOTIFICATION_CONTENT, notificationContent);
        intent.putExtra(COMPLETE_URL, completeUrl);
        intent.putExtra(StaticFields.NOTIFICATION_JSON, json);

        return intent;
    }

    /**
     * Creates an IntentService.  Invoked by your subclass's constructor.
     */
    public NajvaNotificationIntentService() {
        super("Najva IntentService");
    }

    @Override
    protected void onHandleIntent(@Nullable Intent intent) {

        Logger.d("NajvaIntentService", "handling notification click");
        Logger.d("NajvaIntentService", intent.toString());

        //checking if intent is null
        if (intent == null) return;

        // intent has no data to parse
        if (intent.getExtras() == null) return;

        String notificationAction = null;
        String finalAddress = "";
        String url = "";
        int notificationId = 0;
        String uuid = "";

        try {
            notificationAction = intent.getStringExtra(CLICK_ACTION);
            Logger.d("NajvaIntentService", notificationAction);
        } catch (NullPointerException ignored) {
        }

        try {
            finalAddress = intent.getExtras().getString(COMPLETE_URL);
        } catch (NullPointerException ignored) {
        }

        try {
            url = intent.getExtras().getString(FINAL_ADDRESS);
        } catch (NullPointerException ignored){
        }

        try {
            notificationId = intent.getExtras().getInt(NOTIFICATION_ID);
        } catch (NullPointerException ignored) {
        }
        try {
            uuid = intent.getExtras().getString(UUID);
        } catch (NullPointerException ignored) {

        }

        Intent notificationIntent = null;

        assert notificationAction != null;

        ClickReceiverHandler.sendBroadcast(this, uuid);

        switch (notificationAction) {
            case OPEN_LINK: {
                try {
                    url = URLDecoder.decode(url, "UTF-8");
                } catch (UnsupportedEncodingException e) {
                    e.printStackTrace();
                }
                notificationIntent = new Intent(Intent.ACTION_VIEW, Uri.parse(url));
                notificationIntent.putExtra(StaticFields.NOTIFICATION_JSON, intent.getStringExtra(StaticFields.NOTIFICATION_JSON));
                notificationIntent.setFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP | Intent.FLAG_ACTIVITY_SINGLE_TOP | Intent.FLAG_ACTIVITY_NEW_TASK);
                startActivity(notificationIntent);

                break;
            }

            case OPEN_APP: {
                String packageName = getApplicationContext().getPackageName();

                notificationIntent = getApplicationContext().getPackageManager().getLaunchIntentForPackage(packageName);
                if (notificationIntent != null)
                    notificationIntent.putExtra(StaticFields.NOTIFICATION_JSON, intent.getStringExtra(StaticFields.NOTIFICATION_JSON));
                break;
            }

            case OPEN_ACTIVITY: {
                Class<?> act;
                try {
                    act = Class.forName(finalAddress);
                } catch (ClassNotFoundException e) {
                    e.printStackTrace();
                    return;
                }
                notificationIntent = new Intent(getApplicationContext(), act);
                notificationIntent.putExtra(StaticFields.NOTIFICATION_JSON, intent.getStringExtra(StaticFields.NOTIFICATION_JSON));
                break;
            }

            case OPEN_CALL: {
                notificationIntent = new Intent(Intent.ACTION_DIAL, Uri.parse("tel:" + finalAddress));
                break;
            }

            case OPEN_SMS: {
                String content = "";
                try {
                    content = intent.getExtras().getString(NOTIFICATION_CONTENT);
                } catch (NullPointerException ignored) {
                }

                notificationIntent = new Intent(Intent.ACTION_VIEW, Uri.parse("sms:" + finalAddress));
                notificationIntent.putExtra("sms_body", content);
                break;
            }

            case OPEN_TELEGRAM_CHANNEL: {
                notificationIntent = new Intent(Intent.ACTION_VIEW, Uri.parse("tg://resolve?domain=" + finalAddress));
                break;
            }

            case JOIN_TELEGRAM_CHANNEL: {
                notificationIntent = new Intent(Intent.ACTION_VIEW, Uri.parse(finalAddress));
                break;
            }

        }

        if (notificationIntent != null) {

            NotificationManager notificationManager = (NotificationManager) getSystemService(Context.NOTIFICATION_SERVICE);
            assert notificationManager != null;
            notificationManager.cancel(notificationId);

        }

        assert notificationIntent != null;
        notificationIntent.setFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP | Intent.FLAG_ACTIVITY_SINGLE_TOP | Intent.FLAG_ACTIVITY_NEW_TASK);
        startActivity(notificationIntent);

        if (notificationAction.equals(OPEN_LINK)){
            Logger.i("NajvaNotification", "Web links don't require submitting click, skipping");
            return;
        }

        OneTimeWorkRequest oneTimeWorkRequest = new FormRequestWorker.Builder(this)
                .setUrl(ApiInfo.CLICK_LOG.getValue())
                .setMethod(Request.Method.POST)
                .setParams(getClickParams(Uri.parse(finalAddress).getQueryParameter("uuid")))
                .build();
        WorkManager.getInstance(this).enqueue(oneTimeWorkRequest);


        Logger.i("NajvaNotification", "Notification click handled");
    }

    private HashMap<String, String> getClickParams(String uuid) {
        HashMap<String, String> params = new HashMap<>();

        params.put(StaticFields.UUID, uuid);

        String najvaToken = CookieManager.getInstance(this).getCookies();
        if (najvaToken != null) {
            try {
                najvaToken = najvaToken.substring(12, najvaToken.length() - 1);
                params.put(StaticFields.NAJVA_TOKEN, najvaToken);
            } catch (Exception ignored) {
            }
        }

        Logger.d("IntentService", params.toString());

        return params;
    }

}
