package com.najva.sdk.push_notification;

import android.app.Notification;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.graphics.Bitmap;
import android.graphics.Color;
import android.os.Build;
import android.os.PowerManager;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;

import android.util.Log;
import android.widget.ImageView;

import com.android.volley.Response;
import com.android.volley.VolleyError;
import com.android.volley.VolleyLog;
import com.android.volley.toolbox.ImageRequest;
import com.google.firebase.messaging.RemoteMessage;
import com.najva.sdk.ClientMetaData;
import com.najva.sdk.NajvaClient;
import com.najva.sdk.R;
import com.najva.sdk.core.utils.Logger;
import com.najva.sdk.core.utils.Parameter;
import com.najva.sdk.core.utils.StaticFields;
import com.najva.sdk.core.utils.VolleySingleton;
import com.najva.sdk.push_notification.service.NajvaNotificationIntentService;
import com.najva.sdk.runnables.NotificationReceiverHandler;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Map;
import java.util.Random;

public class NotificationUtils {
    private static final String TAG = "NotificationUtils";

    public static NotificationUtils getInstance() {
        return new NotificationUtils();
    }


    private PendingIntent setNotificationAction(@NonNull Context context,
                                                @NonNull String notificationAction,
                                                int notificationId,
                                                @NonNull String finalAddress,
                                                @NonNull String notificationContent,
                                                @NonNull String messageId,
                                                @NonNull String clickType,
                                                int button_id,
                                                @Nullable String json) {

        //replaced with old reflection code
        Intent notificationIntent = NajvaNotificationIntentService.newIntent(context,
                notificationAction,
                notificationId,
                getWebsiteInfo(context),
                finalAddress,
                notificationContent,
                messageId,
                clickType,
                button_id,
                json);

        // generate random integer for pending intent unique
        int dummyUniqueInt = new Random().nextInt(543254);
        return PendingIntent.getService(context, dummyUniqueInt, notificationIntent, PendingIntent.FLAG_ONE_SHOT);

    }

    private ArrayList<String> getWebsiteInfo(@NonNull Context context) {
        ArrayList<String> records = new ArrayList<>();
        records.add(String.valueOf(ClientMetaData.getWebsiteId()));
        records.add(ClientMetaData.getApiKey(context));
        return records;

    }


    public boolean isOreoPlus() {
        return Build.VERSION.SDK_INT >= Build.VERSION_CODES.O;
    }

    public void showNotification(@NonNull final Context context, @NonNull final Map<String, String> data,
                                 @NonNull final String messageId) {
        final String notificationId = data.get(StaticFields.NOTIFICATION_ID);
        final String finalAddress = data.get(StaticFields.URL);
        final String onClickAction = data.get(StaticFields.ONCLICK_ACTION);
        String notificationContent = data.get(StaticFields.NOTIFICATION_CONTENT);
        final String priority = data.get(StaticFields.PRIORITY);

        Logger.d(TAG, data.toString());

        if (notificationContent == null) {
            notificationContent = StaticFields.BLANK;
        }

        assert onClickAction != null;
        assert notificationId != null;
        assert finalAddress != null;


        final PendingIntent pendingIntent = setNotificationAction(
                context,
                onClickAction,
                Integer.parseInt(notificationId),
                finalAddress,
                notificationContent,
                messageId,
                Parameter.ORIGINAL_CLICK_TYPE.getValue(),
                -1,
                data.get(StaticFields.NOTIFICATION_JSON)
        );

        final String title = data.get(StaticFields.TITLE);
        final String body = data.get(StaticFields.BODY);

        assert title != null;
        assert body != null;

        final Notification.Builder notificationBuilder = getNotificationBuilderFactory()
                .newNotificationBuilder(context);
        notificationBuilder
                .setContentTitle(title)
                .setContentText(body)
                .setAutoCancel(true);

        if (body.length() >= 50) {
            Logger.d(TAG, "big text");
            notificationBuilder.setContentText(body);
            notificationBuilder.setStyle(
                    new Notification.BigTextStyle()
                            .setBigContentTitle(title)
            );
        }

        // Since android Oreo notification channel is needed.
        if (isOreoPlus()) {
            assert priority != null;
            if (priority.equalsIgnoreCase(StaticFields.NOTIFICATION_PRIORITY_HIGH)) {
                notificationBuilder.setChannelId(StaticFields.CHANNEL_ID_HIGH_PRIORITY);
            } else {
                notificationBuilder.setChannelId(StaticFields.CHANNEL_ID_LOW_PRIORITY);
            }
        } else {
            updatePriorityUnderOreo(notificationBuilder, priority);
        }


        if (pendingIntent != null) {
            notificationBuilder.setContentIntent(pendingIntent);
        }

        notificationBuilder.setDeleteIntent(getCloseNotificationPendingIntent(context, Integer.parseInt(notificationId)));


        String canLightUp = data.get(StaticFields.LIGHT_UP_SCREEN);

        if (canLightUp != null && canLightUp.equalsIgnoreCase("true")) {
            wakeUpDevice(context);
        }

        if (canDisplayButton()) {
            try {
                JSONArray array = new JSONArray(data.get(StaticFields.BUTTONS));
                for (int i = 0; i < array.length(); i++) {
                    addButton(context, notificationBuilder, Integer.parseInt(notificationId), messageId, array.getJSONObject(i));
                }
            } catch (JSONException e) {
                Logger.d(TAG, "Notification has no buttons");
            }
        }

        String icon = data.get(StaticFields.ICON);
        String image = data.get(StaticFields.IMAGE);

        if (icon == null) {
            if (image != null) {
                downloadImageAndSendNotification(context, notificationBuilder, image,
                        title,
                        body,
                        messageId);
            } else {
                sendNotification(context, notificationBuilder, messageId);
            }
        } else {
            if (image == null) {
                downloadIconAndSendNotification(context, notificationBuilder, icon, messageId);

            } else {
                downloadIconAndImageAndSendNotification(context, notificationBuilder, data, messageId);
            }
        }
    }

    public boolean canDisplayButton() {
        return Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT_WATCH;
    }

    @RequiresApi(api = Build.VERSION_CODES.KITKAT_WATCH)
    public void addButton(Context context, Notification.Builder notificationBuilder, int notificationId, String messageId, JSONObject jsonObject) {
        try {
            String id = jsonObject.getString(StaticFields.ID);
            String title = jsonObject.getString(StaticFields.TITLE);
            String tmpIcon = jsonObject.getString(StaticFields.ICON_NUMBER);
            int icon = IconHandler.getNotificationButtonIcon(Integer.parseInt(tmpIcon));
            String content = jsonObject.getString(StaticFields.CONTENT);
            String onClickAction = jsonObject.getString(StaticFields.ONCLICK_ACTION);
            String finalAddress = jsonObject.getString(StaticFields.FINAL_ADDRESS);

            PendingIntent action = setNotificationAction(
                    context,
                    onClickAction,
                    notificationId,
                    finalAddress,
                    content,
                    messageId,
                    Parameter.BUTTON_CLICK_TYPE.getValue(),
                    Integer.parseInt(id),
                    null
            );

            notificationBuilder.addAction(new Notification.Action(icon, title, action));

        } catch (JSONException e) {
            Logger.d(TAG, "button signature is wrong");
        }
    }

    private PendingIntent getCloseNotificationPendingIntent(Context context, int notificationId) {
        Intent intent = new Intent(StaticFields.ACTION_NOTIFICATION_REMOVED);
        intent.putExtra(StaticFields.NOTIFICATION_ID, notificationId);
        return PendingIntent.getBroadcast(context, notificationId, intent, PendingIntent.FLAG_CANCEL_CURRENT);
    }

    private void downloadIconAndImageAndSendNotification(@NonNull final Context context,
                                                         @NonNull final Notification.Builder notificationBuilder,
                                                         @NonNull final Map<String, String> data,
                                                         @NonNull final String messageId) {

        final String image = data.get(StaticFields.IMAGE);
        final String title = data.get(StaticFields.TITLE);
        final String body = data.get(StaticFields.BODY);
        assert image != null;
        assert title != null;
        assert body != null;

        ImageRequest imageRequest = new ImageRequest(data.get(StaticFields.ICON), new Response.Listener<Bitmap>() {
            @Override
            public void onResponse(Bitmap response) {
                if (response != null) {

                    notificationBuilder.setLargeIcon(response);
                    downloadImageAndSendNotification(context, notificationBuilder
                            , image
                            , title
                            , body
                            , messageId);

                }
            }
        }, 1200, 1200, ImageView.ScaleType.CENTER_CROP, Bitmap.Config.ARGB_8888, new Response.ErrorListener() {
            @Override
            public void onErrorResponse(VolleyError error) {
                VolleyLog.wtf(error.getMessage(), "utf-8");
            }
        });
        VolleySingleton.getInstance(context).add(imageRequest);
    }

    private void downloadIconAndSendNotification(@NonNull final Context context,
                                                 @NonNull final Notification.Builder notificationBuilder,
                                                 @NonNull String icon,
                                                 @NonNull final String messageId) {
        ImageRequest imageRequest = new ImageRequest(icon, new Response.Listener<Bitmap>() {
            @Override
            public void onResponse(Bitmap response) {
                if (response != null) {
                    notificationBuilder.setLargeIcon(response);

                    sendNotification(context, notificationBuilder, messageId);

                }
            }
        }, 1200, 1200, ImageView.ScaleType.CENTER_CROP, Bitmap.Config.ARGB_8888, new Response.ErrorListener() {
            @Override
            public void onErrorResponse(VolleyError error) {
                VolleyLog.wtf(error.getMessage(), "utf-8");
            }
        });

        VolleySingleton.getInstance(context).add(imageRequest);
    }

    public void sendNotification(@NonNull Context context, @NonNull Notification.Builder notificationBuilder,
                                 @NonNull String messageId) {
        int iconRes;
        if (NajvaClient.configuration == null){
            iconRes = R.drawable.najva_notification_icon;
        } else {
            iconRes = NajvaClient.configuration.getNotificationSmallIcon();
        }
        notificationBuilder.setSmallIcon(iconRes);
        notificationBuilder.setDefaults(Notification.DEFAULT_VIBRATE);
        notificationBuilder.setLights(Color.BLUE, 1000, 300);

        NotificationManager notificationManager = (NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE);
        assert notificationManager != null;


        notificationManager.notify(messageId, 0, notificationBuilder.build());

    }

    private void downloadImageAndSendNotification(@NonNull final Context context,
                                                  @NonNull final Notification.Builder notificationBuilder,
                                                  @NonNull String imageUrl,
                                                  @NonNull final String title,
                                                  @NonNull final String body,
                                                  @NonNull final String messageId) {
        ImageRequest imageRequest = new ImageRequest(imageUrl, new Response.Listener<Bitmap>() {
            @Override
            public void onResponse(Bitmap response) {
                if (response != null && Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {

                    Notification.BigPictureStyle bigPictureStyle = new Notification.BigPictureStyle()
                            .bigPicture(response)
                            .setBigContentTitle(title)
                            .setSummaryText(body);

                    notificationBuilder.setStyle(bigPictureStyle);

                    sendNotification(context, notificationBuilder, messageId);
                }
            }
        }, 1200, 1200, ImageView.ScaleType.CENTER_CROP, Bitmap.Config.ARGB_8888, new Response.ErrorListener() {
            @Override
            public void onErrorResponse(VolleyError error) {
                VolleyLog.wtf(error.getMessage(), "utf-8");
                Logger.w(TAG, "Failed to load notification image");
            }
        });
        VolleySingleton.getInstance(context).add(imageRequest);
    }

    private boolean isScreenOn(@NonNull Context context) {
        PowerManager pm = (PowerManager) context.getSystemService(Context.POWER_SERVICE);
        return pm.isScreenOn();
    }

    private void wakeUpDevice(@NonNull Context context) {
        if (!isScreenOn(context)) {
            wakeUp(context);
        }
    }

    private void wakeUp(@NonNull Context context) {
        PowerManager pm = (PowerManager) context.getSystemService(Context.POWER_SERVICE);

        PowerManager.WakeLock wl = pm.newWakeLock(PowerManager.FULL_WAKE_LOCK | PowerManager.ACQUIRE_CAUSES_WAKEUP | PowerManager.ON_AFTER_RELEASE, "najva::wakelog");
        wl.acquire(10000);
        PowerManager.WakeLock wl_cpu = pm.newWakeLock(PowerManager.PARTIAL_WAKE_LOCK, "najva::wakelog");

        wl_cpu.acquire(10000);
    }

    private void updatePriorityUnderOreo(@NonNull Notification.Builder notificationBuilder,
                                         @Nullable String priority) {
        if (priority != null && Build.VERSION.SDK_INT >= 16) {
            if (priority.equalsIgnoreCase(StaticFields.NOTIFICATION_PRIORITY_HIGH)) {
                notificationBuilder.setPriority(Notification.PRIORITY_HIGH);
            } else if (priority.equalsIgnoreCase(StaticFields.NOTIFICATION_PRIORITY_LOW)) {
                notificationBuilder.setPriority(Notification.PRIORITY_LOW);
            }
        }
    }

    public NotificationBuilderFactory getNotificationBuilderFactory() {
        return new NotificationBuilderFactory();
    }

    public boolean isNajvaMessage(@NonNull Context context, @NonNull RemoteMessage remoteMessage) {
        boolean result = false;
        Map<String, String> data = remoteMessage.getData();
        String apiKey = data.get(StaticFields.API_KEY);
        String localApiKey = ClientMetaData.getApiKey(context);

        if (localApiKey == null) {
            Logger.e(TAG, "You must call Najva.initialize() before using isNajvaMessage method");
        } else if (apiKey == null) {
            Logger.e(TAG, "Provide api_key from message is null, contact support!");
        } else {
            result = apiKey.equals(localApiKey);
            if (!result) {
                Logger.e(TAG, "Provide api_key from message is not equal to localApiKey.");
            }
        }

        return result;
    }

    public boolean isMPlus() {
        return Build.VERSION.SDK_INT >= Build.VERSION_CODES.M;
    }

    public static void broadcastNajvaClick(Context context, String messageId, int buttonId) {
        Intent intent = new Intent();
        intent.putExtra(StaticFields.MESSAGE_ID, messageId);
        intent.putExtra(StaticFields.BUTTON_ID, buttonId);
        context.sendBroadcast(intent);
    }

    public void broadcastNajvaNotification(Context context, RemoteMessage remoteMessage) {
        Intent intent = new Intent();
        intent.putExtra(StaticFields.MESSAGE_ID, remoteMessage.getMessageId());
        intent.putExtra(StaticFields.NOTIFICATION_TAG, remoteMessage.getData().get(StaticFields.NOTIFICATION_TAG));
        NotificationReceiverHandler.sendBroadcast(context, intent);
        Log.d(TAG, "broadcastNajvaNotification: " + intent.toString());
    }
}
