package com.najva.sdk;

import android.app.Activity;
import android.app.Application;
import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.content.Context;
import android.content.IntentFilter;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.graphics.Color;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;

import com.najva.sdk.app.AppVersionController;
import com.najva.sdk.app.AppVersionControllerImpl;
import com.najva.sdk.core.ActivityStack;
import com.najva.sdk.core.SessionManager;
import com.najva.sdk.core.ad.AdIdFetchTask;
import com.najva.sdk.core.ad.AdvertisingIdClient;
import com.najva.sdk.core.utils.FileManager;
import com.najva.sdk.core.utils.JsonFile;
import com.najva.sdk.core.utils.Logger;
import com.najva.sdk.core.utils.Parameter;
import com.najva.sdk.core.utils.StaticFields;
//import com.najva.sdk.event_management.TagManager;
//import com.najva.sdk.event_management.view_observer.EventManager;
//import com.najva.sdk.in_app_messaging.InAppMessaging;
import com.najva.sdk.location.LocationController;
import com.najva.sdk.location.LocationControllerImpl;
import com.najva.sdk.login.LoginController;
import com.najva.sdk.login.LoginControllerImpl;
import com.najva.sdk.push_notification.PushNotificationController;
import com.najva.sdk.push_notification.PushNotificationControllerImpl;
import com.najva.sdk.runnables.ClickReceiverHandler;
import com.najva.sdk.runnables.NajvaJsonHandler;
import com.najva.sdk.runnables.NajvaSubscriberTokenHandler;
import com.najva.sdk.runnables.NotificationReceiverHandler;
import com.najva.sdk.utils.MetaDataUtils;

import org.json.JSONArray;
import org.json.JSONException;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;


public class NajvaClient implements Application.ActivityLifecycleCallbacks {

    public static AdvertisingIdClient.AdInfo adInfo;

    public static NajvaConfiguration configuration;

    public static final String TAG = "NajvaClient";

    private static NajvaClient instance;

    public static NajvaClient getInstance(){
        if (instance == null){
            throw new RuntimeException("You must call NajvaClient.getInstance(Context appContext, NajvaConfiguration configuration); first.");
        }
        return instance;
    }

    @NonNull
    public static NajvaClient getInstance(@NonNull Context context,@Nullable NajvaConfiguration configuration){
        if (instance == null){
            if (configuration == null){
                instance = new NajvaClient(context);
            } else {
                instance = new NajvaClient(context, configuration);
            }
        }
        return instance;
    }

    private String apiKey;
    private int websiteId;

    private Context appContext;

    private List<Application.ActivityLifecycleCallbacks> callbacks = new ArrayList<>();

//    private EventManager eventManager;

//    private InAppMessaging inAppMessaging;

    private NajvaClientReceiver receiver;

    private ActivityStack activityStack;

    @Deprecated
    public NajvaClient(Context appContext) {
        this(appContext, new NajvaConfiguration());
    }

    @Deprecated
    public NajvaClient(Context context,NajvaConfiguration configuration) {
        appContext = context;
//        TagManager.init(appContext);
//        eventManager = new EventManager(appContext);
//        inAppMessaging = new InAppMessaging(appContext);
        NajvaClient.configuration = configuration;

        initialize();

        new AdIdFetchTask(appContext).execute();
    }

    @Nullable
    public String getSubscribedToken() {
        String token = FileManager.readFromFile(appContext, Parameter.TOKEN_FILE_NAME.getValue());
        if (token == null || token.isEmpty())
            return null;
        else {
            return token;
        }
    }

    public void getCachedJsonData() {
        if (configuration.hasJsonDataListener()) return;
        JSONArray array = JsonFile.getInstance().readFromJsonFile(appContext);
        if (array == null) return;
        try {
            for (int i = 0; i < array.length(); i++) {
                deliverJsonData(array.getString(i));
            }
            FileManager.deleteCache(appContext, Parameter.NAJVA_CACHE_FILE_NAME.getValue());
        } catch (JSONException e) {
            //e.printStackTrace();
            //write json to file has problem and must be fixed
        }
    }

    private void deliverJsonData(String jsonData) {
        configuration.getNajvaJsonDataListener().onReceiveJson(jsonData);
    }

    private void initialize() {
        initializeLogger();

        initializeManifestMetadata();

        initializeLocation();

        initializeFirebase();

        initializeAppVersion();
//        if (configuration.isScanForTagsEnabled()) {
//            initializeEventManager();
//        }
//        if (configuration.isInAppMessagingEnabled()) {
//            initializeInAppMessaging();
//        }
        initializeBroadCast();
        activityStack = new ActivityStack();
        callbacks.add(activityStack);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            createNotificationChannels();
        }

        Logger.i(TAG,"Najva Initialization complete");
    }

    @RequiresApi(api = Build.VERSION_CODES.O)
    private void createNotificationChannels() {
        NotificationManager mNotificationManager = (NotificationManager) appContext.getSystemService(Context.NOTIFICATION_SERVICE);

        assert mNotificationManager != null;

        NotificationChannel lowChannel = new NotificationChannel(StaticFields.CHANNEL_ID_LOW_PRIORITY,
                StaticFields.CHANNEL_DESCRIPTION_LOW, NotificationManager.IMPORTANCE_DEFAULT);

        lowChannel.setShowBadge(true);
        lowChannel.enableLights(true);
        lowChannel.enableVibration(true);
        lowChannel.setLightColor(Color.CYAN);

        mNotificationManager.createNotificationChannel(lowChannel);

        NotificationChannel highChannel = new NotificationChannel(StaticFields.CHANNEL_ID_HIGH_PRIORITY,
                StaticFields.CHANNEL_DESCRIPTION_HIGH,NotificationManager.IMPORTANCE_HIGH);

        highChannel.setShowBadge(true);
        highChannel.enableLights(true);
        highChannel.enableVibration(true);
        highChannel.setLightColor(Color.CYAN);

        mNotificationManager.createNotificationChannel(highChannel);
    }

    private void initializeLogger() {
        Uri loggerFileUri = Uri.fromFile(appContext.getFilesDir()).buildUpon().appendPath(Parameter.LOGGER_FILE.getValue()).build();
        Logger.setLoggerFilePath(loggerFileUri.getPath());
        Uri indexFileUri = Uri.fromFile(appContext.getFilesDir()).buildUpon().appendPath(Parameter.INDEX_FILE.getValue()).build();
        Logger.setIndexFilePath(indexFileUri.getPath());
        Logger.i(TAG,"Logger initialized");
    }

    private void initializeBroadCast() {
        HashMap<String, NajvaRunnable> actions = new HashMap<>();
        actions.put(StaticFields.EVENT_TYPE_JSON, new NajvaJsonHandler(appContext));
        actions.put(StaticFields.EVENT_TYPE_TOKEN, new NajvaSubscriberTokenHandler());
        actions.put(StaticFields.EVENT_NOTIFICATION_RECEIVER,new NotificationReceiverHandler());
        actions.put(StaticFields.EVENT_CLICK,new ClickReceiverHandler());
//        actions.put(StaticFields.EVENT_TYPE_INAPP, inAppMessaging);
        receiver = new NajvaClientReceiver(actions);
        appContext.registerReceiver(receiver, new IntentFilter(NajvaClientReceiver.ACTION));
    }

    private void initializeInAppMessaging() {
        Logger.i(TAG,"Initializing In-App messaging");
//        callbacks.add(inAppMessaging);
    }

    private void initializeEventManager() {
        Logger.i(TAG,"Initializing event manager");
//        callbacks.add(eventManager);
//        eventManager.load();
    }

    private void initializeAppVersion() {
        AppVersionController appVersionController = new AppVersionControllerImpl(appContext);
        appVersionController.loadAsync();
    }

    private void initializeLogin() {
        LoginController loginController = new LoginControllerImpl(appContext, apiKey, websiteId);
        loginController.loadAsync();
    }

    private void initializeManifestMetadata() {
        apiKey = getApiKey(appContext);
        websiteId = getWebsiteId(appContext);

        ClientMetaData.setWebsiteId(websiteId);
        ClientMetaData.setApiKey(appContext, apiKey);

        Logger.i(TAG,"Manifest metadata has been read");
        Logger.i(TAG,"ApiKey: " + apiKey);
        Logger.i(TAG,"WebsiteId: " +websiteId);
    }

    private void initializeFirebase() {
        if (configuration.isPushNotificationEnabled()) {
            PushNotificationController pushNotificationController = new PushNotificationControllerImpl(appContext,
                    apiKey, websiteId, getSubscribedToken());
            pushNotificationController.loadAsync();

        }
    }

    private void initializeLocation() {
        if (configuration.isLocationEnabled()) {
            LocationController locationController = new LocationControllerImpl(appContext);
            locationController.loadAsync();
            Logger.i(TAG,"Location initialized");
        }
    }

    @Override
    public void onActivityCreated(Activity activity, Bundle savedInstanceState) {
        SessionManager.getInstance().onStart();
        if (activityStack.empty()){
            initializeLogin();
        }
        for (Application.ActivityLifecycleCallbacks callback : callbacks) {
            callback.onActivityCreated(activity, savedInstanceState);
        }
    }


    @Override
    public void onActivityStarted(Activity activity) {
        for (Application.ActivityLifecycleCallbacks callback : callbacks) {
            callback.onActivityStarted(activity);
        }
    }

    @Override
    public void onActivityResumed(Activity activity) {
        for (Application.ActivityLifecycleCallbacks callback : callbacks) {
            callback.onActivityResumed(activity);
        }
    }

    @Override
    public void onActivityPaused(Activity activity) {
        for (Application.ActivityLifecycleCallbacks callback : callbacks) {
            callback.onActivityPaused(activity);
        }
    }

    @Override
    public void onActivityStopped(Activity activity) {
        for (Application.ActivityLifecycleCallbacks callback : callbacks) {
            callback.onActivityStopped(activity);
        }
    }

    @Override
    public void onActivitySaveInstanceState(Activity activity, Bundle outState) {
        for (Application.ActivityLifecycleCallbacks callback : callbacks) {
            callback.onActivitySaveInstanceState(activity, outState);
        }
    }

    @Override
    public void onActivityDestroyed(Activity activity) {
        if (activityStack.isEmpty()) {
            SessionManager.getInstance().onStop();
        }
        for (Application.ActivityLifecycleCallbacks callback : callbacks) {
            callback.onActivityDestroyed(activity);
        }
    }

    public void onAppTerminated() {
        appContext.unregisterReceiver(receiver);
    }

    private int getWebsiteId(@NonNull Context context) {
        return MetaDataUtils.getInstance(getApplicationInfo(context)).getWebsiteId();
    }

    private String getApiKey(@NonNull Context context) {
        return MetaDataUtils.getInstance(getApplicationInfo(context)).getApiKey();
    }

    private ApplicationInfo getApplicationInfo(@NonNull Context context) {
        try {
            return context.getPackageManager().getApplicationInfo(context.getPackageName(), PackageManager.GET_META_DATA);
        } catch (PackageManager.NameNotFoundException e) {
            e.printStackTrace();
        }
        return null;
    }

}
