package com.najva.sdk.event_management.view_observer;

import android.app.Activity;
import android.util.Log;
import android.view.View;

import androidx.fragment.app.Fragment;
import androidx.fragment.app.FragmentActivity;
import androidx.recyclerview.widget.RecyclerView;

import com.najva.sdk.event_management.TagManager;
import com.najva.sdk.event_management.ViewUtils;
import com.najva.sdk.event_management.adapters.WrapperAdapter;
import com.najva.sdk.event_management.clickListener.OnClickListener;
import com.najva.sdk.event_management.model.ActivityRule;
import com.najva.sdk.event_management.model.FragmentRule;
import com.najva.sdk.event_management.model.ListRule;
import com.najva.sdk.event_management.model.Rule;

import java.util.List;


class ViewManager {
    private TagManager tagManager;
    private FragmentViewManager fragmentViewManager;

    ViewManager(TagManager tagManager) {
        this.tagManager = tagManager;
        fragmentViewManager = new FragmentViewManager(this);
    }

    void handleViewEventListener(Rule rule, View rootView) {
        int id = ViewUtils.getId(EventManager.RIdClass, rule.getItemId());
        if (id > 0) {
            Log.d("EventManager", rule.toString());
            Log.d("EventManager", "view with id " + rule.getItemId() + " found with int id: " + id);

            View view = rootView.findViewById(id);
            View.OnClickListener listener = null;

            if (view.hasOnClickListeners()) {
                Log.d("EventManager", "view already has click listener");

                listener = ViewUtils.getOnClickListenerForView(view);
            }

            view.setOnClickListener(new OnClickListener(rule, tagManager, listener));
        }
    }

    void handleListEventListener(ListRule rule, View rootView) {
        int id = ViewUtils.getListId(EventManager.RIdClass, rule.getName());

        Log.d("EventManager", "handling " + rule.toString());

        RecyclerView recyclerView = rootView.findViewById(id);
        if (recyclerView == null) return;
        RecyclerView.Adapter adapter = recyclerView.getAdapter();

        if (adapter != null) {
            Log.d("EventManager", "setting adapter for " + rule.getRules().get(0).toString());

            WrapperAdapter wrapperAdapter = new WrapperAdapter(adapter, tagManager);
            for (Rule r : rule.getRules()) {
                wrapperAdapter.addRule(r);
            }
            recyclerView.setAdapter(wrapperAdapter);
            wrapperAdapter.notifyDataSetChanged();
        }
    }

    void handleActivity(ActivityRule activityRule, Activity activity) {
        for (Rule rule : activityRule.getRules()) {
            Log.d("ViewManager", "handling direct items in activity: " + rule.toString());
            handleViewEventListener(rule, activity.findViewById(android.R.id.content));
        }
        for (ListRule rule : activityRule.getListRules()) {
            Log.d("ViewManager", "handling list in activity: " + rule.toString());
            handleListEventListener(rule, activity.findViewById(android.R.id.content));
        }

        if (!activityRule.getFragmentRules().isEmpty()) {
            Log.d("ViewManager", "handling fragment in activity: ");
            handleFragments(activity, activityRule.getFragmentRules());
        }
    }

    private void handleFragments(Activity activity, List<FragmentRule> fragmentRules) {
        if (activity instanceof FragmentActivity) {
            fragmentViewManager.handleFragment((FragmentActivity) activity, fragmentRules);
        }
    }

    void onActivityDestroyed(Activity activity) {
        if (activity instanceof FragmentActivity) {
            fragmentViewManager.unregisterActivityFragments((FragmentActivity) activity);
        }
    }

    void registerFragmentCallback(Activity activity) {
        if (activity instanceof FragmentActivity) {
            fragmentViewManager.registerLifecycleCallback((FragmentActivity) activity);
        }
    }
}
