package com.najva.sdk.event_management.view_observer;

import android.app.Activity;
import android.content.Context;
import android.os.Bundle;
import android.util.Log;
import android.view.View;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.fragment.app.Fragment;
import androidx.fragment.app.FragmentActivity;
import androidx.fragment.app.FragmentManager;

import com.najva.sdk.event_management.model.FragmentRule;
import com.najva.sdk.event_management.model.ListRule;
import com.najva.sdk.event_management.model.Rule;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class FragmentViewManager extends FragmentManager.FragmentLifecycleCallbacks {

    private ViewManager viewManager;

    private Map<FragmentActivity, List<FragmentRule>> fragmentRuleMap = new HashMap<>();

    private List<Fragment> attachedFragments = new ArrayList<>();

    FragmentViewManager(ViewManager viewManager) {

        this.viewManager = viewManager;
    }

    void handleFragment(FragmentActivity activity, List<FragmentRule> fragmentRule) {
        fragmentRuleMap.put(activity, fragmentRule);

        for (Fragment fragment : attachedFragments) {
            handleFragmentEvents(fragment);
        }
    }

    void unregisterActivityFragments(FragmentActivity activity) {
        activity.getSupportFragmentManager().unregisterFragmentLifecycleCallbacks(this);
    }

    @Override
    public void onFragmentAttached(@NonNull FragmentManager fm, @NonNull Fragment f, @NonNull Context context) {
        super.onFragmentAttached(fm, f, context);
        attachedFragments.add(f);
    }

    @Override
    public void onFragmentDetached(@NonNull FragmentManager fm, @NonNull Fragment f) {
        super.onFragmentDetached(fm, f);
        attachedFragments.remove(f);
    }

    @Override
    public void onFragmentViewCreated(@NonNull FragmentManager fm, @NonNull Fragment f,
                                      @NonNull View v, @Nullable Bundle savedInstanceState) {
        super.onFragmentViewCreated(fm, f, v, savedInstanceState);

        handleFragmentEvents(f);
    }

    private void handleFragmentEvents(Fragment f) {
        View view = f.getView();

        Log.d("FragmentViewManager", "onFragmentCreated: called");

        if (view == null) {
            Log.d("FragmentViewManager", "fragment view is null");
            return;
        }

        List<FragmentRule> rules = fragmentRuleMap.get(f.getActivity());

        if (rules == null) {
            Log.d("FragmentViewManager", "fragmentRule is null");
            return;
        }

        FragmentRule rule = findRuleForFragment(rules, f);
        if (rule == null) {
            Log.d("FragmentViewManager", "no rule found for fragment " + f.getClass().getSimpleName());
            return;
        }
        for (ListRule listRule : rule.getListRules()) {
            Log.d("FragmentViewManager", "list rule");
            viewManager.handleListEventListener(listRule, view);
        }

        for (Rule r : rule.getRules()) {
            Log.d("FragmentViewManager", "item rule");
            viewManager.handleViewEventListener(r, view);
        }
    }

    private FragmentRule findRuleForFragment(List<FragmentRule> rules, Fragment f) {
        for (FragmentRule rule : rules) {
            Log.d("FragmentEventManager", f.getClass().getSimpleName());
            if (rule.getName().equals(f.getClass().getSimpleName().replace(".", ""))) {
                return rule;
            }
        }
        return null;
    }

    void registerLifecycleCallback(FragmentActivity activity) {
        Log.d("FragmentViewManager", "registering callback for fragment lifecycle");
        activity.getSupportFragmentManager().registerFragmentLifecycleCallbacks(this, true);
    }
}
