package com.najva.sdk.event_management.database;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;
import android.util.Log;

import androidx.annotation.Nullable;

import com.najva.sdk.event_management.model.Rule;
import com.najva.sdk.event_management.model.Tag;

import java.util.ArrayList;
import java.util.List;

public class TagDatabase extends SQLiteOpenHelper {

    private static final String DB_NAME = "najva-database";

    private static final String TAG_TABLE_NAME = "tags";
    private static final String TAG_ID = "id";
    private static final String TAG_NAME = "name";
    private static final String TAG_OPERATOR = "operator";

    private static final String RULE_TABLE_NAME = "rules";
    private static final String RULE_ID = "id";
    private static final String RULE_TYPE = "type";
    private static final String RULE_ITEM_ID = "item_id";
    private static final String RULE_ACTION = "action";
    private static final String RULE_ACTIVITY_NAME = "activity_name";
    private static final String RULE_TAG_ID = "tag_id";
    private static final String RULE_SATISFIED = "satisfied";
    private static final String RULE_FRAGMENT = "fragment_name";
    private static final String RULE_LIST = "list_id";

    public TagDatabase(@Nullable Context context) {
        super(context, DB_NAME, null, 1);
    }

    @Override
    public void onCreate(SQLiteDatabase db) {
        String createTagTableQuery = String.format("create table if not exists %s (" + // table name
                        " %s INTEGER PRIMARY KEY," + // id
                        " %s TEXT," + // name
                        " %s TEXT);" // operator
                , TAG_TABLE_NAME, TAG_ID, TAG_NAME, TAG_OPERATOR);
        String createRuleTableQuery = String.format("create table if not exists %s (" + // tabel name
                        " %s INTEGER PRIMARY KEY," + // id
                        " %s TEXT," + // type
                        " %s TEXT," + // item_id
                        " %s TEXT," + // action
                        " %s TEXT," + // activity
                        " %s TEXT," + // fragment
                        " %s TEXT," + //listId
                        " %s INTEGER," + // tag_id
                        " %s INTEGER);",
                RULE_TABLE_NAME, RULE_ID, RULE_TYPE, RULE_ITEM_ID, RULE_ACTION, RULE_ACTIVITY_NAME,
                RULE_FRAGMENT, RULE_LIST, RULE_TAG_ID, RULE_SATISFIED);

        Log.d("TagDatabase", createTagTableQuery);
        Log.d("TagDatabase", createRuleTableQuery);

        db.execSQL(createTagTableQuery);
        db.execSQL(createRuleTableQuery);
    }

    @Override
    public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
        // noting to implement.
    }

    public void insertTag(Tag tag) {
        ContentValues values = new ContentValues();
        values.put(TAG_ID, tag.getId());
        values.put(TAG_NAME, tag.getName());
        values.put(TAG_OPERATOR, tag.getOperator());

        getWritableDatabase().insert(TAG_TABLE_NAME, null, values);
    }

    public void removeTag(Tag tag) {
        getWritableDatabase().delete(TAG_TABLE_NAME, TAG_ID + "=" + tag.getId(), null);
        getWritableDatabase().delete(RULE_TABLE_NAME, RULE_TAG_ID + "=" + tag.getId(), null);
    }

    public void addRuleForTag(Tag tag, Rule rule) {
        ContentValues values = new ContentValues();
        values.put(RULE_ID, rule.getId());
        values.put(RULE_TAG_ID, tag.getId());
        values.put(RULE_ACTION, rule.getAction());
        values.put(RULE_TYPE, rule.getType());
        values.put(RULE_FRAGMENT, rule.getFragmentName());
        values.put(RULE_LIST, rule.getListId());
        values.put(RULE_ACTIVITY_NAME, rule.getActivityName());
        values.put(RULE_ITEM_ID, rule.getItemId());
        values.put(RULE_SATISFIED, rule.getSatisfied());

        getWritableDatabase().insert(RULE_TABLE_NAME, null, values);
    }

    public void updateRule(Rule rule) {
        ContentValues values = new ContentValues();
        values.put(RULE_ACTION, rule.getAction());
        values.put(RULE_TYPE, rule.getType());
        values.put(RULE_ACTIVITY_NAME, rule.getActivityName());
        values.put(RULE_ITEM_ID, rule.getItemId());
        values.put(RULE_FRAGMENT, rule.getFragmentName());
        values.put(RULE_LIST, rule.getListId());
        values.put(RULE_SATISFIED, rule.getSatisfied());

        getWritableDatabase().update(RULE_TABLE_NAME, values, RULE_ID + "=" + rule.getId(), null);
    }

    public List<Tag> getTags() {
        Cursor tagCursor = getReadableDatabase().rawQuery("select * from " + TAG_TABLE_NAME + " ;", null);

        List<Tag> tags = new ArrayList<>();

        loadTagsFromCursor(tagCursor, tags);

        tagCursor.close();

        return tags;
    }

    private void loadTagsFromCursor(Cursor tagCursor, List<Tag> tags) {
        if (tagCursor.getCount() == 0) return;
        if (!tagCursor.moveToFirst()) return;
        do {
            Tag tag = parseTagFromCursor(tagCursor);
            tags.add(tag);
        } while (tagCursor.moveToNext());
    }

    private Tag parseTagFromCursor(Cursor tagCursor) {

//        Log.d("TagDatabase", Arrays.toString(tagCursor.getColumnNames()));
//        Log.d("TagDatabase", tagCursor.getColumnCount() + "");
        Tag tag = new Tag(tagCursor.getInt(tagCursor.getColumnIndex(TAG_ID)));
        tag.setOperator(tagCursor.getString(tagCursor.getColumnIndex(TAG_OPERATOR)));
        tag.setName(tagCursor.getString(tagCursor.getColumnIndex(TAG_NAME)));

        tag.setRules(getRulesForTag(tag.getId()));

        return tag;
    }

    private List<Rule> getRulesForTag(int id) {
        Cursor ruleCursor = getReadableDatabase().rawQuery("select * from " + RULE_TABLE_NAME +
                " where " + RULE_TAG_ID + "=" + id + " ;", null);

        List<Rule> rules = new ArrayList<>();

        loadRules(ruleCursor, rules);

        ruleCursor.close();

        return rules;
    }

    List<Rule> getRules() {
        Cursor cursor = getReadableDatabase().rawQuery("select * from " + RULE_TABLE_NAME + ";", null);

        List<Rule> rules = new ArrayList<>();

        loadRules(cursor, rules);

        cursor.close();

        return rules;
    }

    private void loadRules(Cursor ruleCursor, List<Rule> rules) {
        if (ruleCursor.getCount() == 0) return;
        if (!ruleCursor.moveToFirst()) return;

        do {
            Rule rule = parseRuleFromCursor(ruleCursor);
            rules.add(rule);
        } while (ruleCursor.moveToNext());
    }

    private Rule parseRuleFromCursor(Cursor ruleCursor) {
//        Log.d("Database", ruleCursor.getString(ruleCursor.getColumnIndex(RULE_ACTIVITY_NAME)));

        Rule rule = new Rule(ruleCursor.getInt(ruleCursor.getColumnIndex(RULE_ID)));
        rule.setItemId(ruleCursor.getString(ruleCursor.getColumnIndex(RULE_ITEM_ID)));
        rule.setActivityName(ruleCursor.getString(ruleCursor.getColumnIndex(RULE_ACTIVITY_NAME)));
        rule.setAction(ruleCursor.getString(ruleCursor.getColumnIndex(RULE_ACTION)));
        rule.setType(ruleCursor.getString(ruleCursor.getColumnIndex(RULE_TYPE)));
        rule.setFragmentName(ruleCursor.getString(ruleCursor.getColumnIndex(RULE_FRAGMENT)));
        rule.setListId(ruleCursor.getString(ruleCursor.getColumnIndex(RULE_LIST)));
        rule.setSatisfied(ruleCursor.getInt(ruleCursor.getColumnIndex(RULE_SATISFIED)) == 1);

        return rule;
    }

    public void clean() {
        getWritableDatabase().delete(TAG_TABLE_NAME, "1", null);
        getWritableDatabase().delete(RULE_TABLE_NAME, "1", null);
    }

    public void removeRule(Rule rule) {
        getWritableDatabase().delete(RULE_TABLE_NAME, RULE_ID + "=" + rule.getId(), null);
    }
}
