package com.najva.sdk.core.works;

import android.content.Context;
import android.content.SharedPreferences;

import androidx.annotation.NonNull;

import android.util.Log;

import androidx.work.BackoffPolicy;
import androidx.work.Constraints;
import androidx.work.Data;
import androidx.work.NetworkType;
import androidx.work.OneTimeWorkRequest;
import androidx.work.Worker;
import androidx.work.WorkerParameters;

import com.android.volley.Request;
import com.android.volley.VolleyError;
import com.najva.sdk.core.utils.CookieManager;
import com.najva.sdk.core.utils.IBuilder;
import com.najva.sdk.core.utils.MetaJsonRequest;
import com.najva.sdk.core.utils.StaticFields;
import com.najva.sdk.core.utils.VolleyErrorHandler;
import com.najva.sdk.core.utils.VolleySingleton;
import com.najva.sdk.core.utils.WorkManager;

import org.json.JSONObject;

import java.io.File;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

public class JsonRequestWorker extends Worker {

    private static final String TAG = "JsonRequestWorker";
    private MetaJsonRequest request;
    private String name;

    public JsonRequestWorker(@NonNull Context context, @NonNull WorkerParameters workerParams) {
        super(context, workerParams);
    }

    private void initiateRequest() {
        name = getInputData().getString("name");
        this.request = new MetaJsonRequest.Builder()
                .setAsync(false)
                .setCookieManager(CookieManager.getInstance(getApplicationContext()))
                .setUrl(getUrl())
                .setParams(getPostParams())
                .setHeaders(getHeaders())
                .setMethod(getMethod())
                .build();
        Log.d(TAG, "params: " + getPostParams().toString());
        Log.d(TAG, "header: " + getHeaders().toString());
        Log.d(TAG, "method: " + getMethod());
    }

    private Map<String, String> getPostParams() {
        Map<String, String> params = new HashMap<>();
        SharedPreferences workerMap = getApplicationContext().getSharedPreferences(StaticFields.POST + name, Context.MODE_PRIVATE);
        for (String key : workerMap.getAll().keySet()) {
            params.put(key, workerMap.getString(key, ""));
        }
        return params;
    }

    private Map<String, String> getHeaders() {
        Map<String, String> headers = new HashMap<>();
        SharedPreferences workerMap = getApplicationContext().getSharedPreferences(StaticFields.HEADER + name, Context.MODE_PRIVATE);
        for (String key : workerMap.getAll().keySet()) {
            headers.put(key, workerMap.getString(key, ""));

        }
        return headers;
    }

    private int getMethod() {
        return getInputData().getInt(StaticFields.METHOD, Request.Method.GET);
    }

    private String getUrl() {
        return getInputData().getString(StaticFields.URL);
    }

    @NonNull
    @Override
    public Result doWork() {
        initiateRequest();
        try {
            VolleySingleton.getInstance(getApplicationContext()).add(request);
            JSONObject response = request.getFuture().get();
            deleteSharePreferencesFiles();
            Log.d(TAG, "deleting shared_preferences files for: " + name);
            return Result.success(new Data.Builder().putString("response", response.toString()).build());
        }
        // Interrupt exception is for threading problems
        catch (InterruptedException e) {
            return Result.retry();
        }
        // error is now for volley
        catch (ExecutionException e) {
            if (e.getCause() instanceof VolleyError) {
                Result result = VolleyErrorHandler.handleError((VolleyError) e.getCause());
                if (result.equals(Result.failure())) {
                    request.getFuture().onErrorResponse((VolleyError) e.getCause());
                }
                if (!(result instanceof Result.Retry)) {
                    deleteSharePreferencesFiles();
                    Log.d(TAG, "deleting shared_preferences files for: " + name);
                }
                return result;
            }
            return Result.retry();
        }
        // timeout error
        catch (AssertionError e) {
            return Result.retry();
        }
    }

    private void deleteSharePreferencesFiles() {
        new File(getApplicationContext().getFilesDir().getParent()
                + "/shared_prefs/" + "post" + name + ".xml").delete();
        new File(getApplicationContext().getFilesDir().getParent()
                + "/shared_prefs" + "header" + name + ".xml").delete();
    }

    public static class Builder implements IBuilder<OneTimeWorkRequest> {

        private String url;
        private int method = Request.Method.GET;
        private SharedPreferences params;
        private SharedPreferences headers;
        String name;

        public Builder(@NonNull Context context) {
            // random name
            // when using a single sharedPref for requests it may cause problems like
            // when you have two or more workers at same time and this causes conflicts
            // to the params and headers.
            name = generateRandom();

            Log.d(TAG, "sahredPreferencesName: " + name);

            params = context.getSharedPreferences(StaticFields.POST + name, Context.MODE_PRIVATE);
            headers = context.getSharedPreferences(StaticFields.HEADER + name, Context.MODE_PRIVATE);
        }

        @NonNull
        private String generateRandom() {
            return UUID.randomUUID().toString().replace("-", "");
        }

        public Builder setUrl(@NonNull String url) {
            this.url = url;
            return this;
        }

        public Builder setMethod(int method) {
            this.method = method;
            return this;
        }

        public Builder addParam(@NonNull String key, @NonNull String value) {
            params.edit().putString(key, value).apply();
            return this;
        }

        public Builder addHeader(@NonNull String key, @NonNull String value) {
            headers.edit().putString(key, value).apply();
            return this;
        }

        public Builder setParams(@NonNull Map<String, String> params) {
            SharedPreferences.Editor editor = this.params.edit();
            for (String key : params.keySet()) {
                editor.putString(key, params.get(key));
            }
            editor.apply();
            return this;
        }

        public Builder setHeaders(@NonNull Map<String, String> headers) {
            SharedPreferences.Editor editor = this.headers.edit();
            for (String key : headers.keySet()) {
                editor.putString(key, headers.get(key));
            }
            editor.apply();
            return this;
        }


        @NonNull
        @Override
        public OneTimeWorkRequest build() {
            Constraints constraints = new Constraints.Builder()
                    .setRequiredNetworkType(NetworkType.CONNECTED)
                    .build();

            Data.Builder dataBuilder = new Data.Builder();

            dataBuilder.putString(StaticFields.URL, url);
            dataBuilder.putInt(StaticFields.METHOD, method);
            dataBuilder.putString(StaticFields.NAME, name);

            return new OneTimeWorkRequest.Builder(JsonRequestWorker.class)
                    .setInputData(dataBuilder.build()).setConstraints(constraints)
                    .setBackoffCriteria(BackoffPolicy.EXPONENTIAL, 1, TimeUnit.MINUTES)
                    .addTag(WorkManager.TAG)
                    .addTag(TAG)
                    .build();
        }
    }
}
