package com.najva.sdk;

import android.app.Activity;
import android.app.Application;
import android.content.Context;
import android.content.IntentFilter;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.os.Bundle;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.najva.sdk.app.AppVersionController;
import com.najva.sdk.app.AppVersionControllerImpl;
import com.najva.sdk.core.utils.FileManager;
import com.najva.sdk.core.utils.JsonFile;
import com.najva.sdk.core.utils.Parameter;
import com.najva.sdk.core.utils.StaticFields;
import com.najva.sdk.crash_report.NajvaCrashReporter;
import com.najva.sdk.event_management.TagManager;
import com.najva.sdk.event_management.view_observer.EventManager;
import com.najva.sdk.in_app_messaging.InAppMessenger;
import com.najva.sdk.location.LocationController;
import com.najva.sdk.location.LocationControllerImpl;
import com.najva.sdk.login.LoginController;
import com.najva.sdk.login.LoginControllerImpl;
import com.najva.sdk.push_notification.PushNotificationController;
import com.najva.sdk.push_notification.PushNotificationControllerImpl;
import com.najva.sdk.runnables.NajvaJsonHandler;
import com.najva.sdk.runnables.NajvaSubscriberTokenHandler;
import com.najva.sdk.utils.MetaDataUtils;

import org.json.JSONArray;
import org.json.JSONException;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

public class NajvaClient implements Application.ActivityLifecycleCallbacks {

    private NajvaConfiguration configuration = new NajvaConfiguration();

    private String apiKey;
    private int websiteId;

    private Context appContext;

    private List<Application.ActivityLifecycleCallbacks> callbacks = new ArrayList<>();

    private EventManager eventManager;

    private InAppMessenger inAppMessenger;

    private NajvaClientReceiver receiver;

    public NajvaClient(Context appContext){
        this(appContext,new NajvaConfiguration());
    }

    public NajvaClient(Context appContext,NajvaConfiguration configuration) {
        this.appContext = appContext;
        TagManager tagManager = new TagManager(appContext);
        eventManager = new EventManager(appContext, tagManager);
        inAppMessenger = new InAppMessenger(appContext);
        this.configuration = configuration;

        initialize();
    }

    @Nullable
    public String getSubscribedToken() {
        String token = FileManager.readFromFile(appContext, Parameter.TOKEN_FILE_NAME.getValue());
        if (token == null || token.isEmpty())
            return null;
        else {
            return token;
        }
    }

    public void getCachedJsonData() {
        if (configuration.hasJsonDataListener()) return;
        JSONArray array = JsonFile.getInstance().readFromJsonFile(appContext);
        if (array == null) return;
        try {
            for (int i = 0; i < array.length(); i++) {
                deliverJsonData(array.getString(i));
            }
            FileManager.deleteCache(appContext, Parameter.NAJVA_CACHE_FILE_NAME.getValue());
        } catch (JSONException e) {
            //e.printStackTrace();
            //write json to file has problem and must be fixed
        }
    }

    private void deliverJsonData(String jsonData) {
        configuration.getNajvaJsonDataListener().onReceiveJson(jsonData);
    }

    private void initialize() {
        initializeManifestMetadata();
        if (configuration.isLocationEnabled()) {
            initializeLocation();
        }
        initializeFirebase();

        initializeLogin();

        initializeAppVersion();
//        if (configuration.isScanForTagsEnabled()) {
//            initializeEventManager();
//        }
//        if (configuration.isInAppMessagingEnabled()) {
//            initializeInAppMessaging();
//        }
        initializeBroadCast();
    }

    private void initializeBroadCast() {
        HashMap<String, NajvaRunnable> actions = new HashMap<>();
        actions.put(StaticFields.EVENT_TYPE_JSON, new NajvaJsonHandler(appContext, configuration.getNajvaJsonDataListener()));
        actions.put(StaticFields.EVENT_TYPE_TOKEN, new NajvaSubscriberTokenHandler(configuration.getUserSubscriptionListener()));
        actions.put(StaticFields.EVENT_TYPE_INAPP, inAppMessenger);
        receiver = new NajvaClientReceiver(actions);
        appContext.registerReceiver(receiver, new IntentFilter(NajvaClientReceiver.ACTION));
    }

    private void initializeInAppMessaging() {
//        if (configuration.isInAppMessagingEnabled()) {
//            callbacks.add(inAppMessenger);
//        }
    }

    private void initializeEventManager() {
//        if (configuration.isScanForTagsEnabled()) {
//            callbacks.add(eventManager);
//        }
    }

    private void initializeAppVersion() {
        AppVersionController appVersionController = new AppVersionControllerImpl(appContext);
        appVersionController.loadAsync();
    }

    private void initializeLogin() {
        LoginController loginController = new LoginControllerImpl(appContext, apiKey, websiteId);
        loginController.loadAsync();
    }

    private void initializeCrashReporter() {
        NajvaCrashReporter.getInstance(appContext).loadAsync();
    }

    private void initializeManifestMetadata() {
        apiKey = getApiKey(appContext);
        websiteId = getWebsiteId(appContext);

        ClientMetaData.setWebsiteId(websiteId);
        ClientMetaData.setApiKey(appContext, apiKey);

    }

    private void initializeFirebase() {
        if (configuration.isPushNotificationEnabled()) {
            PushNotificationController pushNotificationController = new PushNotificationControllerImpl(appContext,
                    apiKey, websiteId, getSubscribedToken());
            pushNotificationController.loadAsync();
        }
    }

    private void initializeLocation() {
        if (configuration.isLocationEnabled()) {
            LocationController locationController = new LocationControllerImpl(appContext);
            locationController.loadAsync();
        }
    }

    @Override
    public void onActivityCreated(Activity activity, Bundle savedInstanceState) {
        for (Application.ActivityLifecycleCallbacks callback : callbacks) {
            callback.onActivityCreated(activity, savedInstanceState);
        }
    }


    @Override
    public void onActivityStarted(Activity activity) {
        for (Application.ActivityLifecycleCallbacks callback : callbacks) {
            callback.onActivityStarted(activity);
        }
    }

    @Override
    public void onActivityResumed(Activity activity) {
        for (Application.ActivityLifecycleCallbacks callback : callbacks) {
            callback.onActivityResumed(activity);
        }
    }

    @Override
    public void onActivityPaused(Activity activity) {
        for (Application.ActivityLifecycleCallbacks callback : callbacks) {
            callback.onActivityPaused(activity);
        }
    }

    @Override
    public void onActivityStopped(Activity activity) {
        for (Application.ActivityLifecycleCallbacks callback : callbacks) {
            callback.onActivityStopped(activity);
        }
    }

    @Override
    public void onActivitySaveInstanceState(Activity activity, Bundle outState) {
        for (Application.ActivityLifecycleCallbacks callback : callbacks) {
            callback.onActivitySaveInstanceState(activity, outState);
        }
    }

    @Override
    public void onActivityDestroyed(Activity activity) {
        for (Application.ActivityLifecycleCallbacks callback : callbacks) {
            callback.onActivityDestroyed(activity);
        }
    }

    public void onAppTerminated() {
        appContext.unregisterReceiver(receiver);
    }

    private int getWebsiteId(@NonNull Context context) {
        return MetaDataUtils.getInstance(getApplicationInfo(context)).getWebsiteId();
    }

    private String getApiKey(@NonNull Context context) {
        return MetaDataUtils.getInstance(getApplicationInfo(context)).getApiKey();
    }

    private ApplicationInfo getApplicationInfo(@NonNull Context context) {
        try {
            return context.getPackageManager().getApplicationInfo(context.getPackageName(), PackageManager.GET_META_DATA);
        } catch (PackageManager.NameNotFoundException e) {
            e.printStackTrace();
        }
        return null;
    }

}
