package com.najva.sdk.location;

import android.content.Context;
import android.location.Location;
import android.location.LocationListener;
import android.location.LocationManager;
import android.os.Bundle;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.google.android.gms.common.internal.Preconditions;
import com.najva.sdk.core.utils.Logger;
import com.najva.sdk.utils.DeviceUtils;

import java.math.BigDecimal;

import static android.Manifest.permission.ACCESS_COARSE_LOCATION;
import static android.Manifest.permission.ACCESS_FINE_LOCATION;

/**
 * A helper class for using android location api.
 * This class use GPS and Network Location
 * These permission is mandatory for using this class
 * - GPS mandatory permissions : ACCESS_FINE_LOCATION
 * - NETWORK_LOCATION mandatory permissions: ACCESS_FINE_LOCATION and ACCESS_COARSE_LOCATION
 * <p>
 * For getting device location use {@link #getLastKnownLocation(Context, int, LocationAwareness)}
 */
public class LocationManagerApiWrapper implements LocationListener {
    private static final int MIN_TIME = 1000;
    private static final int MIN_DISTANCE = 100;
    private static final int PRECISION = 8;
    public static final String TAG = "LocationApiWrapper";


    private static volatile LocationManagerApiWrapper instance;

    private Location location;

    public static LocationManagerApiWrapper getInstance(@NonNull Context context) {
        if (instance == null) {
            synchronized (LocationManagerApiWrapper.class) {
                if (instance == null) {
                    instance = new LocationManagerApiWrapper(context);
                }
            }
        }
        return instance;
    }

    private LocationManagerApiWrapper(@NonNull Context context) {
        this.location = getLastKnownLocation(context, 0, LocationAwareness.NORMAL);

        final LocationManager locationManager =
                (LocationManager) context.getSystemService(Context.LOCATION_SERVICE);


        if (ValidLocationProvider.GPS.hasRequiredPermissions(context)) {
            locationManager.requestLocationUpdates(
                    LocationManager.GPS_PROVIDER,
                    MIN_TIME,
                    MIN_DISTANCE,
                    this
            );
        } else if (ValidLocationProvider.NETWORK.hasRequiredPermissions(context)) {
            locationManager.requestLocationUpdates(
                    LocationManager.NETWORK_PROVIDER,
                    MIN_TIME,
                    MIN_DISTANCE,
                    this
            );
        } else {
            Logger.d(TAG, "GPS or Network permission does not granted");
        }
    }

    public Location getTruncatedLocation(@NonNull Context context) {
        if (ValidLocationProvider.GPS.hasRequiredPermissions(context) || ValidLocationProvider.NETWORK.hasRequiredPermissions(context)) {
            truncateLocationLatLon(location, PRECISION);
            return location;
        } else {
            Logger.d(TAG, "GPS or Network permission does not granted");
            return null;
        }
    }

    public Location getLocation(@NonNull Context context) {
        if (hasGPSPermission(context) || hasNetworkPermission(context)) {
            return location;
        } else {
            Logger.d(TAG, "GPS or Network permission does not granted");
            return null;
        }
    }

    public boolean hasGPSPermission(Context context) {
        return ValidLocationProvider.GPS.hasRequiredPermissions(context);
    }

    public boolean hasNetworkPermission(Context context) {
        return ValidLocationProvider.NETWORK.hasRequiredPermissions(context);
    }

    @Override
    public void onLocationChanged(Location location) {
        this.location = location;
    }

    @Override
    public void onStatusChanged(String provider, int status, Bundle extras) {
        Logger.d(TAG, "Location status changed, " + status);
    }

    @Override
    public void onProviderEnabled(String provider) {
        Logger.i(TAG, "Location provider enabled");
    }

    @Override
    public void onProviderDisabled(String provider) {
        Logger.d(TAG, "Location provide disabled");

    }

    public enum ValidLocationProvider {
        NETWORK(LocationManager.NETWORK_PROVIDER),
        GPS(LocationManager.GPS_PROVIDER);

        @NonNull
        final String name;

        ValidLocationProvider(@NonNull final String name) {
            this.name = name;
        }

        @NonNull
        @Override
        public String toString() {
            return name;
        }

        private boolean hasRequiredPermissions(@NonNull final Context context) {
            switch (this) {
                case NETWORK:
                    return DeviceUtils.isPermissionGranted(context, ACCESS_FINE_LOCATION)
                            || DeviceUtils.isPermissionGranted(context, ACCESS_COARSE_LOCATION);
                case GPS:
                    return DeviceUtils.isPermissionGranted(context, ACCESS_FINE_LOCATION);
                default:
                    return false;
            }
        }
    }

    /**
     * Returns the last known location of the device using its GPS and network location providers.
     * May be null if:
     * - Location permissions are not requested in the Android manifest file
     * - The location providers don't exist
     *
     * @return instance of {@link Location} that contains longitude and latitude
     */
    @Nullable
    private static Location getLastKnownLocation(@NonNull final Context context,
                                                 final int locationPrecision,
                                                 final @NonNull LocationAwareness locationAwareness) {
        Preconditions.checkNotNull(context);
        Preconditions.checkNotNull(locationAwareness);

        if (locationAwareness == LocationAwareness.DISABLED) {
            return null;
        }

        final Location gpsLocation = getLocationFromProvider(context,
                ValidLocationProvider.GPS);
        final Location networkLocation = getLocationFromProvider(context,
                ValidLocationProvider.NETWORK);

        if (gpsLocation == null && networkLocation == null) {
            return null;
        }

        final Location result = getMostRecentValidLocation(gpsLocation, networkLocation);


        // Truncate latitude/longitude to the number of digits specified by locationPrecision.
        if (locationAwareness == LocationAwareness.TRUNCATED) {
            truncateLocationLatLon(result, locationPrecision);
        }

        return result;
    }

    @Nullable
    private static Location getLocationFromProvider(@NonNull final Context context,
                                                    @NonNull final ValidLocationProvider provider) {

        Location location = null;

        Preconditions.checkNotNull(context);
        Preconditions.checkNotNull(provider);

        if (provider.hasRequiredPermissions(context)) {
            final LocationManager locationManager =
                    (LocationManager) context.getSystemService(Context.LOCATION_SERVICE);
            try {
                // noinspection ResourceType
                location = locationManager.getLastKnownLocation(provider.toString());
            } catch (SecurityException e) {
                Logger.d(TAG, "Failed to retrieve location from " +
                        provider.toString() + " provider: access appears to be disabled.");
            } catch (IllegalArgumentException | NullPointerException e) {
                Logger.d(TAG, "Failed to retrieve location: device has no " +
                        provider.toString() + " location provider.");
            } catch (Exception e) {
                Logger.d(TAG, "Unknown error occurred when getLocationFromProvider, " + (e.getMessage() != null ? e.getMessage() : ""));
            }

        } else {
            Logger.d(TAG, "GPS or Network permission does not granted");
        }

        return location;
    }

    @Nullable
    private static Location getMostRecentValidLocation(@Nullable final Location a,
                                                       @Nullable final Location b) {
        if (a == null) {
            return b;
        }

        if (b == null) {
            return a;
        }

        // At this point, locations A and B are non-null, so return the more recent one
        return (a.getTime() > b.getTime()) ? a : b;
    }

    private static void truncateLocationLatLon(@Nullable final Location location,
                                               final int precision) {
        if (location == null || precision < 0) {
            return;
        }

        double lat = location.getLatitude();
        double truncatedLat = BigDecimal.valueOf(lat)
                .setScale(precision, BigDecimal.ROUND_HALF_DOWN)
                .doubleValue();
        location.setLatitude(truncatedLat);

        double lon = location.getLongitude();
        double truncatedLon = BigDecimal.valueOf(lon)
                .setScale(precision, BigDecimal.ROUND_HALF_DOWN)
                .doubleValue();
        location.setLongitude(truncatedLon);
    }
}
