package com.najva.sdk.in_app_messaging;

import android.app.Activity;
import android.app.Application;
import android.content.Context;
import android.content.Intent;
import android.graphics.Color;
import android.os.Bundle;
import android.os.Handler;


import com.android.volley.VolleyError;
import com.najva.sdk.NajvaClientReceiver;
import com.najva.sdk.NajvaRunnable;
import com.najva.sdk.core.task.BaseAsyncTask;
import com.najva.sdk.core.utils.Logger;
import com.najva.sdk.core.utils.StaticFields;
import com.najva.sdk.in_app_messaging.database.InAppDeleteTask;
import com.najva.sdk.in_app_messaging.database.InAppInsertTask;
import com.najva.sdk.in_app_messaging.database.InAppReadTask;
import com.najva.sdk.in_app_messaging.database.InAppRepository;
import com.najva.sdk.in_app_messaging.model.ButtonProperty;
import com.najva.sdk.in_app_messaging.model.CloseButton;
import com.najva.sdk.in_app_messaging.model.CloseButtonType;
import com.najva.sdk.in_app_messaging.model.ImageProperty;
import com.najva.sdk.in_app_messaging.model.InApp;
import com.najva.sdk.in_app_messaging.model.TextProperty;
import com.najva.sdk.in_app_messaging.presenter.BannerPresenter;
import com.najva.sdk.in_app_messaging.view.InAppActivity;
import com.najva.sdk.in_app_messaging.view.InAppType;

import java.util.ArrayList;
import java.util.GregorianCalendar;
import java.util.List;

public class InAppMessaging implements Application.ActivityLifecycleCallbacks, NajvaRunnable, InAppMessagingCallBack {
    private Activity currentActivity;
    public static final String TAG = "InAppMessaging";
    private Context appContext;
    private List<InApp> inAppList = new ArrayList<>();
    private long startTime;
    private BannerPresenter bannerPresenter;
    private boolean isInAppExist = false;

    public static void sendBroadcast(Context context, Intent intent) {
        Intent inAppIntent = new Intent();
        inAppIntent.setAction(NajvaClientReceiver.ACTION);
        inAppIntent.putExtra(StaticFields.ACTION, StaticFields.EVENT_TYPE_INAPP);
        inAppIntent.putExtras(intent);
        context.sendBroadcast(inAppIntent);
    }

    public InAppMessaging(Context appContext) {
        this.appContext = appContext;

        new InAppReadTask(InAppRepository.getInstance(appContext), new BaseAsyncTask.AsyncTaskPostExecuteListener<List<InApp>>() {
            @Override
            public void onResult(final List<InApp> result) {
                if (result.size() != 0) {
                    inAppList = result;
                }
            }
        }).execute();
    }

    @Override
    public void onActivityCreated(Activity activity, Bundle savedInstanceState) {
    }


    @Override
    public void onActivityStarted(Activity activity) {

    }

    @Override
    public void onActivityResumed(Activity activity) {

        currentActivity = activity;
        if (activity instanceof InAppActivity)
            return;


        if (inAppList.size() != 0) {

            processInAppMessage(inAppList.get(0));
            deleteInAppsFromDB();

        }

        if (bannerPresenter != null) {
            bannerPresenter.onActivityResumed(activity);
        }

    }

    @Override
    public void onActivityPaused(Activity activity) {
        if (bannerPresenter != null) {
            bannerPresenter.onActivityPaused();
        }
        currentActivity = null;
    }

    @Override
    public void onActivityStopped(Activity activity) {
    }

    @Override
    public void onActivitySaveInstanceState(Activity activity, Bundle outState) {

    }

    @Override
    public void onActivityDestroyed(Activity activity) {

    }

    @Override
    public void run(Intent data) {

        processInAppMessage(parseInApp(data));

    }

    private void processInAppMessage(final InApp inApp) {
        Logger.d(TAG, "displayInAppMessage: " + currentActivity);

        if (currentActivity == null || isInAppExist) {
            saveInApp(inApp);
        } else {
            isInAppExist = true;
            if (System.currentTimeMillis() > inApp.getEndDate().getTime()) {
                return;
            }

            startTime = System.currentTimeMillis();
            NajvaImageLoader.getInstance(currentActivity)
                    .load(inApp.getImage().getImageUrl())
                    .into(new NajvaImageLoader.OnCompleteLoadImage() {
                        @Override
                        public void onSuccess() {
                            long loadDelay = System.currentTimeMillis() - startTime;
                            displayInApp(loadDelay, inApp);
                        }

                        @Override
                        public void onFailure(VolleyError error, String url) {
                            if (inApp.isShowInLowConnection()) {
                                long loadDelay = System.currentTimeMillis() - startTime;
                                displayInApp(loadDelay, inApp);
                            }
                        }
                    });
        }
    }

    private void saveInApp(InApp inApp) {
        new InAppInsertTask(InAppRepository.getInstance(appContext)).execute(inApp);
        inAppList.add(inApp);
    }

    private void displayInApp(long loadDelay, final InApp inApp) {
        if (loadDelay < inApp.getDelay())
            new Handler().postDelayed(new Runnable() {
                @Override
                public void run() {
                    if (inApp.getType() == InAppType.HEADER) {
                        bannerPresenter = BannerPresenter.getInstance(currentActivity, inApp, InAppMessaging.this);
                        bannerPresenter.handleInApp();
                    } else {
                        if (currentActivity != null) {
                            currentActivity.startActivity(InAppActivity.newIntent(currentActivity, inApp, InAppMessaging.this));
                        }
                    }
                }
            }, inApp.getDelay() - loadDelay);
        else {
            if (inApp.getType() == InAppType.HEADER) {
                bannerPresenter = BannerPresenter.getInstance(currentActivity, inApp, InAppMessaging.this);
                bannerPresenter.handleInApp();
            } else {

                if (currentActivity != null) {
                    currentActivity.startActivity(InAppActivity.newIntent(currentActivity, inApp, InAppMessaging.this));
                }
            }
        }
    }

    private InApp parseInApp(Intent data) {
        // parse json to inAppModel
        boolean titleAvailable = false;
        boolean desAvailable = false;
        boolean priButton = false;
        boolean secButton = false;
        InAppType type = InAppType.IMAGE_ONLY;
        Logger.d(TAG, "parseInApp: " + data.getExtras().toString());
        if (data.getExtras().getString("type").equals("modal")) {
            titleAvailable = true;
            desAvailable = true;
            priButton = true;
            secButton = true;
            type = InAppType.MODAL;
        } else if (data.getExtras().getString("type").equals("image")) {
        } else if (data.getExtras().getString("type").equals("banner")) {
            titleAvailable = true;
            desAvailable = false;
            priButton = true;
            secButton = true;
            type = InAppType.HEADER;
        } else {
            titleAvailable = true;
            desAvailable = false;
            priButton = true;
            secButton = false;
            type = InAppType.FULL_SCREEN;
        }

        TextProperty title = new TextProperty();
        title.setAvailable(titleAvailable);
        title.setText(data.getExtras().getString("title"));
        title.setVisibility(!data.getExtras().getString("title").equals(""));
        title.setColor(Color.parseColor("#EEEEEE"));

        TextProperty description = new TextProperty();
        description.setAvailable(desAvailable);
        description.setText(data.getExtras().getString("des"));
        description.setVisibility(!data.getExtras().getString("des").equals(""));
        description.setColor(Color.parseColor("#EEEEEE"));

        ButtonProperty primaryButton = new ButtonProperty();
        primaryButton.setAvailable(priButton);
        primaryButton.setText(data.getExtras().getString("primary"));
        primaryButton.setAction("https://google.com");
        primaryButton.setVisibility(true);
        primaryButton.setTextColor(Color.parseColor("#EEEEEE"));
        primaryButton.setBackgroundColor(Color.parseColor("#424242"));

        ButtonProperty secondaryButton = new ButtonProperty();
        secondaryButton.setAvailable(secButton);
        secondaryButton.setText(data.getExtras().getString("secondary"));
        secondaryButton.setAction("https://najva.com");
        secondaryButton.setVisibility(!data.getExtras().getString("secondary").equals(""));
        secondaryButton.setTextColor(Color.parseColor("#EEEEEE"));
        secondaryButton.setBackgroundColor(Color.parseColor("#424242"));

        ImageProperty image = new ImageProperty();
        image.setVisibility(!data.getExtras().getString("url").equals(""));
        image.setAction("https://google.com");
        image.setImageUrl(data.getExtras().getString("url"));

        CloseButton closeButton = new CloseButton();
        closeButton.setColor(Color.parseColor("#00B0FF"));
        closeButton.setType(CloseButtonType.CLOSE_WINDOW);

        return InApp.InAppBuilder.anInApp()
                .withTitle(title)
                .withType(type)
                .withDescription(description)
                .withButtonPrimary(primaryButton)
                .withButtonSecondary(secondaryButton)
                .withImage(image)
                .withShowInLowConnection(true)
                .withEndDate(new GregorianCalendar(2022, 2, 7, 12, 10, 20).getTime())
                .withBackgroundColor(Color.parseColor("#212121"))
                .withDelay(3000)
                .withCloseButton(closeButton)
                .build();
    }

    private void deleteInAppsFromDB() {
        inAppList.clear();
        new InAppDeleteTask(InAppRepository.getInstance(appContext)).execute();
    }

    @Override
    public void inAppWasShown() {
        isInAppExist = false;
    }
}
