package com.najva.sdk.push_notification.service;

import android.app.IntentService;
import android.app.NotificationManager;
import android.content.Context;
import android.content.Intent;
import android.net.Uri;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;


import androidx.work.OneTimeWorkRequest;
import androidx.work.WorkManager;

import com.android.volley.Request;
import com.najva.sdk.core.utils.ApiInfo;
import com.najva.sdk.core.utils.CookieManager;
import com.najva.sdk.core.utils.Logger;
import com.najva.sdk.core.utils.Parameter;
import com.najva.sdk.core.utils.StaticFields;
import com.najva.sdk.core.works.FormRequestWorker;
import com.najva.sdk.push_notification.NotificationUtils;
import com.najva.sdk.runnables.ClickReceiverHandler;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.HashMap;

public class NajvaNotificationIntentService extends IntentService {

    private static final String MESSAGE_ID = "message_id";
    private static final String CLICK_ACTION = "onclick_action";
    private static final String FINAL_ADDRESS = "final_address";
    private static final String NOTIFICATION_ID = "notification_id";
    private static final String WEBSITE_INFO = "website_info";
    private static final String NOTIFICATION_CONTENT = "notification_content";
    private static final String CLICK_TYPE = "click_type";
    private static final String BUTTON_ID = "button_id";

    private static final String OPEN_LINK = "open-link";
    public static final String OPEN_APP = "open-app";
    private static final String OPEN_ACTIVITY = "open-activity";
    private static final String OPEN_CALL = "open-call";
    private static final String OPEN_SMS = "open-sms";
    private static final String OPEN_TELEGRAM_CHANNEL = "open-telegram-channel";
    private static final String JOIN_TELEGRAM_CHANNEL = "join-telegram-channel";

    public static Intent newIntent(@NonNull Context context, @NonNull String notificationAction,
                                   int notificationId,
                                   ArrayList<String> websiteInfo,
                                   String finalAddress,
                                   String notificationContent,
                                   String messageId,
                                   String clickType,
                                   int button_id,
                                   String json) {

        Intent intent = new Intent(context, NajvaNotificationIntentService.class);
        intent.putExtra(MESSAGE_ID, messageId);
        intent.putExtra(CLICK_ACTION, notificationAction);
        intent.putExtra(FINAL_ADDRESS, finalAddress);
        intent.putExtra(NOTIFICATION_ID, notificationId);
        intent.putExtra(WEBSITE_INFO, websiteInfo);
        intent.putExtra(NOTIFICATION_CONTENT, notificationContent);
        intent.putExtra(StaticFields.NOTIFICATION_JSON, json);
        // click on notification or on a button of notification?
        intent.putExtra(CLICK_TYPE, clickType);
        // button-id
        if (button_id > 0) {
            intent.putExtra(BUTTON_ID, button_id);
        }

        return intent;
    }

    /**
     * Creates an IntentService.  Invoked by your subclass's constructor.
     */
    public NajvaNotificationIntentService() {
        super("Najva IntentService");
    }

    @Override
    protected void onHandleIntent(@Nullable Intent intent) {

        Logger.d("NajvaIntentService","handling notification click");

        //checking if intent is null
        if (intent == null) return;

        // intent has no data to parse
        if (intent.getExtras() == null) return;

        String notificationAction = null;
        String finalAddress = "";
        int notificationId = 0;
        String messageId = "";
        String clickType = null;
        int buttonId = -1;

        try {
            buttonId = intent.getExtras().getInt(BUTTON_ID);
        } catch (NullPointerException ignored) {
        }

        try {
            clickType = intent.getExtras().getString(CLICK_TYPE);
        } catch (NullPointerException ignored) {
        }

        try {
            messageId = intent.getExtras().getString(MESSAGE_ID);
        } catch (NullPointerException ignored) {
        }

        try {
            notificationAction = intent.getStringExtra(CLICK_ACTION);
        } catch (NullPointerException ignored) {
        }

        try {
            finalAddress = intent.getExtras().getString(FINAL_ADDRESS);
        } catch (NullPointerException ignored) {
        }

        try {
            notificationId = intent.getExtras().getInt(NOTIFICATION_ID);
        } catch (NullPointerException ignored) {
        }


        final ArrayList<String> websiteInfo = intent.getExtras().getStringArrayList(WEBSITE_INFO);

        Intent notificationIntent = null;

        assert notificationAction != null;

        ClickReceiverHandler.sendBroadcast(this, messageId, buttonId);

        switch (notificationAction) {
            case OPEN_LINK: {
                try {
                    finalAddress = URLDecoder.decode(finalAddress, "UTF-8");
                } catch (UnsupportedEncodingException e) {
                    e.printStackTrace();
                }
//                String url = ApiInfo.REDIRECT_CLICK.getValue() + "?notification_id=" + notificationId + "&website_id=" + websiteInfo.get(0) + "&api_key=" + websiteInfo.get(1) + "&next=" + finalAddress;
                String url = finalAddress;
                notificationIntent = new Intent(Intent.ACTION_VIEW, Uri.parse(url));
                notificationIntent.putExtra(StaticFields.NOTIFICATION_JSON, intent.getStringExtra(StaticFields.NOTIFICATION_JSON));
                notificationIntent.setFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP | Intent.FLAG_ACTIVITY_SINGLE_TOP | Intent.FLAG_ACTIVITY_NEW_TASK);
                startActivity(notificationIntent);

                break;
            }

            case OPEN_APP: {
                String packageName = getApplicationContext().getPackageName();

                notificationIntent = getApplicationContext().getPackageManager().getLaunchIntentForPackage(packageName);
                if (notificationIntent != null)
                    notificationIntent.putExtra(StaticFields.NOTIFICATION_JSON, intent.getStringExtra(StaticFields.NOTIFICATION_JSON));
                break;
            }

            case OPEN_ACTIVITY: {
                Class<?> act;
                try {
                    act = Class.forName(finalAddress);
                } catch (ClassNotFoundException e) {
                    e.printStackTrace();
                    return;
                }
                notificationIntent = new Intent(getApplicationContext(), act);
                notificationIntent.putExtra(StaticFields.NOTIFICATION_JSON, intent.getStringExtra(StaticFields.NOTIFICATION_JSON));
                break;
            }

            case OPEN_CALL: {
                notificationIntent = new Intent(Intent.ACTION_DIAL, Uri.parse("tel:" + finalAddress));
                break;
            }

            case OPEN_SMS: {
                String content = "";
                try {
                    content = intent.getExtras().getString(NOTIFICATION_CONTENT);
                } catch (NullPointerException ignored) {
                }

                notificationIntent = new Intent(Intent.ACTION_VIEW, Uri.parse("sms:" + finalAddress));
                notificationIntent.putExtra("sms_body", content);
                break;
            }

            case OPEN_TELEGRAM_CHANNEL: {
                notificationIntent = new Intent(Intent.ACTION_VIEW, Uri.parse("tg://resolve?domain=" + finalAddress));
                break;
            }

            case JOIN_TELEGRAM_CHANNEL: {
                notificationIntent = new Intent(Intent.ACTION_VIEW, Uri.parse(finalAddress));
                break;
            }

        }

        if (notificationIntent != null) {
            if (clickType != null && clickType.equalsIgnoreCase(Parameter.BUTTON_CLICK_TYPE.getValue())) {
                NotificationManager notificationManager = (NotificationManager) getSystemService(Context.NOTIFICATION_SERVICE);
                assert notificationManager != null;
                notificationManager.cancel(messageId, 0);
            }
        }

        assert notificationIntent != null;
        notificationIntent.setFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP | Intent.FLAG_ACTIVITY_SINGLE_TOP | Intent.FLAG_ACTIVITY_NEW_TASK);
        startActivity(notificationIntent);


        if (clickType != null && clickType.equalsIgnoreCase(Parameter.ORIGINAL_CLICK_TYPE.getValue())) {

            OneTimeWorkRequest oneTimeWorkRequest = new FormRequestWorker.Builder(this)
                    .setUrl(ApiInfo.CLICK_LOG.getValue())
                    .setMethod(Request.Method.POST)
                    .setParams(getClickParams(websiteInfo.get(1), websiteInfo.get(0), notificationId))
                    .build();
            WorkManager.getInstance(this).enqueue(oneTimeWorkRequest);

        } else if (clickType != null && buttonId > 0 && clickType.equalsIgnoreCase(Parameter.BUTTON_CLICK_TYPE.getValue())) {

            OneTimeWorkRequest oneTimeWorkRequest = new FormRequestWorker.Builder(this)
                    .setUrl(ApiInfo.BUTTON_CLICK_LOG.getValue())
                    .setParams(getButtonClickLog(websiteInfo.get(1), websiteInfo.get(0), notificationId, buttonId))
                    .setMethod(Request.Method.POST)
                    .build();
            WorkManager.getInstance(this).enqueue(oneTimeWorkRequest);

            NotificationManager manager = (NotificationManager) getSystemService(NOTIFICATION_SERVICE);
            manager.cancel(notificationId);

        }

        Logger.i("NajvaNotification","Notification click handled");
//
    }

    private HashMap<String, String> getClickParams(@NonNull String apiKey, @NonNull String websiteId,
                                                   int notificationId) {
        HashMap<String, String> params = new HashMap<>();

        params.put(StaticFields.WEBSITE_ID, websiteId);
        params.put(StaticFields.API_KEY, apiKey);
        params.put(StaticFields.NOTIFICATION_ID, String.valueOf(notificationId));

        String najvaToken = CookieManager.getInstance(this).getCookies();
        if (najvaToken != null) {
            try {
                najvaToken = najvaToken.substring(12, najvaToken.length() - 1);
                params.put(StaticFields.NAJVA_TOKEN, najvaToken);
            } catch (Exception ignored) {
            }
        }

        return params;
    }

    private HashMap<String, String> getButtonClickLog(@NonNull String apiKey, @NonNull String websiteId,
                                                      int notificationId, int buttonId) {
        HashMap<String, String> params = new HashMap<>();

        params.put(StaticFields.WEBSITE_ID, websiteId);
        params.put(StaticFields.API_KEY, apiKey);
        params.put(StaticFields.NOTIFICATION_ID, String.valueOf(notificationId));
        params.put(StaticFields.BUTTON_ID, String.valueOf(buttonId));

        return params;
    }

}
