package com.najva.sdk.push_notification.handlers;

import android.app.Notification;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.graphics.drawable.AdaptiveIconDrawable;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.os.PowerManager;

import androidx.annotation.DrawableRes;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;

import com.najva.sdk.ClientMetaData;
import com.najva.sdk.R;
import com.najva.sdk.core.utils.Logger;
import com.najva.sdk.core.utils.Parameter;
import com.najva.sdk.core.utils.StaticFields;
import com.najva.sdk.push_notification.IconHandler;
import com.najva.sdk.push_notification.NotificationBuilderFactory;
import com.najva.sdk.push_notification.service.NajvaNotificationIntentService;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Map;
import java.util.Random;

public abstract class BaseNotificationHandler {
    final Context context;
    final Map<String, String> data;
    final String messageId;

    Notification.Builder builder;

    private NotificationBuilderFactory factory;

    public BaseNotificationHandler(Context context, Map<String, String> data, String messageId) {

        this.context = context;
        this.data = data;
        this.messageId = messageId;
        factory = new NotificationBuilderFactory();
    }

    public abstract void updateBuilder(Notification.Builder builder);

    @RequiresApi(api = Build.VERSION_CODES.O)
    private Notification.Builder createBuilder() {
        return factory.newNotificationBuilder(context, getNotificationChannel());
    }

    String getNotificationChannel() {
        String channel = data.get(StaticFields.CHANNEL);
        if (channel != null) {
            return channel;
        }
        String priority = data.get(StaticFields.PRIORITY);
        if (priority != null && priority.equalsIgnoreCase(StaticFields.NOTIFICATION_PRIORITY_LOW)) {
            return StaticFields.CHANNEL_ID_LOW_PRIORITY;
        } else {
            return StaticFields.CHANNEL_ID_HIGH_PRIORITY;
        }
    }

    private NotificationManager getNotificationManager() {
        return (NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE);
    }

    @RequiresApi(api = Build.VERSION_CODES.O)
    public void showNotification() {

        builder = createBuilder();
        setNotificationIntents(builder);
        setButtons(builder);

        wakeUpDevice();

        updateBuilder(builder);
    }

    void sendNotification() {
        getNotificationManager().notify(0, builder.build());
    }

    @RequiresApi(api = Build.VERSION_CODES.KITKAT_WATCH)
    private void setButtons(Notification.Builder builder) {
        try {
            JSONArray array = new JSONArray(data.get(StaticFields.BUTTONS));
            for (int i = 0; i < array.length(); i++) {
                addButton(builder, array.getJSONObject(i));
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @RequiresApi(api = Build.VERSION_CODES.KITKAT_WATCH)
    private void addButton(Notification.Builder builder, JSONObject jsonObject) {
        try {
            String id = jsonObject.getString(StaticFields.ID);
            String title = jsonObject.getString(StaticFields.TITLE);
            String tmpIcon = jsonObject.getString(StaticFields.ICON_NUMBER);
            int icon = IconHandler.getNotificationButtonIcon(Integer.parseInt(tmpIcon));

            PendingIntent action = setNotificationAction(
                    Parameter.BUTTON_CLICK_TYPE.getValue(),
                    Integer.parseInt(id),
                    null
            );

            builder.addAction(new Notification.Action(icon, title, action));
            Logger.d("NotificationHandler", "adding button");
        } catch (JSONException ignored) {
            ignored.printStackTrace();
        }
    }

    private void setNotificationIntents(Notification.Builder builder) {
        PendingIntent clickIntent = setNotificationAction(Parameter.ORIGINAL_CLICK_TYPE.getValue()
                , -1, data.get(StaticFields.NOTIFICATION_JSON));
        builder.setContentIntent(clickIntent);

        PendingIntent deleteIntent = getCloseNotificationPendingIntent(
                Integer.parseInt(data.get(StaticFields.NOTIFICATION_ID))
        );
        builder.setDeleteIntent(deleteIntent);

    }

    /**
     * gets application icon resource id
     *
     * @return default icon if any exception throws
     */
    @DrawableRes
    int getApplicationIcon() {
        PackageManager pm = context.getPackageManager();
        try {
            ApplicationInfo info = pm.getApplicationInfo(context.getPackageName(), PackageManager.GET_META_DATA);
            return info.icon;
        } catch (PackageManager.NameNotFoundException e) {
            return R.drawable.ic_notification_icon;
        }
    }

    /**
     * gets foreground icon of application
     *
     * @return null if foreground icon is not available
     */
    @RequiresApi(api = Build.VERSION_CODES.O)
    Drawable getForegroundIcon() {
        PackageManager pm = context.getPackageManager();
        try {
            ApplicationInfo info = pm.getApplicationInfo(context.getPackageName(), PackageManager.GET_META_DATA);
            int resId = info.icon;
            Drawable drawable = context.getDrawable(resId);
            if (drawable instanceof AdaptiveIconDrawable) {
                return ((AdaptiveIconDrawable) drawable).getForeground();
            }
        } catch (PackageManager.NameNotFoundException e) {
            return null;
        }
        return null;
    }

    private PendingIntent setNotificationAction(@NonNull String clickType,
                                                int button_id,
                                                @Nullable String json) {

        String notificationContent = data.get(StaticFields.NOTIFICATION_CONTENT);
        if (notificationContent == null) {
            notificationContent = StaticFields.BLANK;
        }
        Intent notificationIntent = NajvaNotificationIntentService.newIntent(context,
                data.get(StaticFields.ONCLICK_ACTION),
                Integer.valueOf(data.get(StaticFields.NOTIFICATION_ID)),
                getWebsiteInfo(context),
                data.get(StaticFields.URL),
                notificationContent,
                messageId,
                clickType,
                button_id,
                json);

        int dummyUniqueInt = new Random().nextInt(543254);
        return PendingIntent.getService(context, dummyUniqueInt, notificationIntent, PendingIntent.FLAG_ONE_SHOT);
    }

    private PendingIntent getCloseNotificationPendingIntent(int notificationId) {
        Intent intent = new Intent(StaticFields.ACTION_NOTIFICATION_REMOVED);
        intent.putExtra(StaticFields.NOTIFICATION_ID, notificationId);
        return PendingIntent.getBroadcast(context, notificationId, intent, PendingIntent.FLAG_CANCEL_CURRENT);
    }

    private ArrayList<String> getWebsiteInfo(@NonNull Context context) {
        ArrayList<String> records = new ArrayList<>();
        records.add(String.valueOf(ClientMetaData.getWebsiteId()));
        records.add(ClientMetaData.getApiKey(context));
        return records;
    }

    private boolean isScreenOn() {
        PowerManager pm = (PowerManager) context.getSystemService(Context.POWER_SERVICE);
        return pm.isScreenOn();
    }

    private void wakeUpDevice() {
        if (!isScreenOn()) {
            wakeUp();
        }
    }

    private void wakeUp() {
        PowerManager pm = (PowerManager) context.getSystemService(Context.POWER_SERVICE);

        PowerManager.WakeLock wl = pm.newWakeLock(PowerManager.FULL_WAKE_LOCK | PowerManager.ACQUIRE_CAUSES_WAKEUP | PowerManager.ON_AFTER_RELEASE, "najva::wakelog");
        wl.acquire(10000);
        PowerManager.WakeLock wl_cpu = pm.newWakeLock(PowerManager.PARTIAL_WAKE_LOCK, "najva::wakelog");

        wl_cpu.acquire(10000);
    }

}
