package com.najva.sdk.push_notification;

import android.content.Context;
import android.content.Intent;
import android.os.Handler;
import android.os.Looper;
import android.util.Log;

import androidx.annotation.NonNull;

import androidx.work.OneTimeWorkRequest;
import androidx.work.WorkManager;

import com.android.volley.Request;
import com.google.android.gms.tasks.OnSuccessListener;
import com.google.firebase.FirebaseApp;
import com.google.firebase.iid.FirebaseInstanceId;
import com.google.firebase.iid.InstanceIdResult;
import com.google.firebase.messaging.RemoteMessage;
import com.najva.sdk.ClientMetaData;
import com.najva.sdk.Najva;
import com.najva.sdk.NajvaClient;
import com.najva.sdk.campaign.CampaignController;
import com.najva.sdk.campaign.CampaignControllerImpl;
import com.najva.sdk.core.utils.ApiInfo;
import com.najva.sdk.core.utils.FileManager;
import com.najva.sdk.core.utils.Logger;
import com.najva.sdk.core.utils.NotificationActionHandler;
import com.najva.sdk.core.utils.Parameter;
import com.najva.sdk.core.utils.StaticFields;
import com.najva.sdk.core.works.FormRequestWorker;
//import com.najva.sdk.in_app_messaging.InAppMessaging;
import com.najva.sdk.location.LocationControllerImpl;
import com.najva.sdk.remote_config.RemoteConfigControllerImpl;
import com.najva.sdk.runnables.NajvaJsonHandler;


import java.util.Map;

public class NajvaPushNotificationHandler {
    private static final String TAG = "NajvaPushNotifHandler";

    public static void handleMessage(@NonNull final Context context, @NonNull RemoteMessage remoteMessage) {
        NotificationUtils notificationUtils = NotificationUtils.getInstance();
        sendNotification(context, remoteMessage, notificationUtils);
    }

    // for better testing no item should be initialized during image_placeholder so we have to take them as parameters
    public static void sendNotification(@NonNull final Context context, @NonNull final RemoteMessage remoteMessage,
                                        @NonNull NotificationUtils notificationUtils) {
        String messageId = remoteMessage.getMessageId();
        final Map<String, String> data = remoteMessage.getData();
        Logger.i(TAG, "new notification received");

        if (notificationUtils.isNajvaMessage(context, remoteMessage)) {
            String notificationType = getNotificationType(data.get(StaticFields.NOTIFICATION_TYPE));
            if (notificationType.equals(StaticFields.JSON_DATA)) {

                sendJson(context, data);

            } else if (notificationType.equals(StaticFields.DATA_CONFIG)) {
                //sdk config
                Logger.d(TAG, "sendNotification>>>sdk config: ");
                NotificationActionHandler.handleNajvaConfig(context, data, messageId);
            } else if (notificationType.equals(StaticFields.IN_APP_MESSAGING)) {
                Intent inAppIntent = new Intent();
                inAppIntent.putExtra("in-app", data.get("in-app"));
//                InAppMessaging.sendBroadcast(context,inAppIntent);

            } else {

                assert messageId != null;
                Logger.d(TAG, "sendNotification>>>simpleNotification: " + data.toString());

                notificationUtils.showNotification(context.getApplicationContext(), data, messageId);
                notificationUtils.broadcastNajvaNotification(context, remoteMessage);
                //sendNotificationReceiveLogToServer(context, remoteMessage);
            }

        }
    }

    private static void sendJson(Context context, final Map<String, String> data) {
        final String jsonData = data.get(StaticFields.JSON_DATA);
        Logger.i(TAG, "sendNotification>>>jsonData: " + jsonData);
        if (jsonData != null) {
            sendJson(context, jsonData);
        }
    }

    private static void sendJson(final Context context, final String jsonData) {
        if (!jsonData.isEmpty()) {
            new Handler(Looper.getMainLooper()).post(new Runnable() {
                public void run() {
                    Intent intent = new Intent();
                    intent.putExtra(Najva.JSON_DATA, jsonData);
                    NajvaJsonHandler.sendBroadcast(context, intent);
                }
            });
        }
    }

    private static void sendNotificationReceiveLogToServer(Context context, RemoteMessage remoteMessage) {
        String notificationId = remoteMessage.getData().get(StaticFields.NOTIFICATION_ID);
        OneTimeWorkRequest request = new FormRequestWorker.Builder(context)
                .addParam(StaticFields.NOTIFICATION_ID, notificationId)
                .addParam(StaticFields.WEBSITE_ID, String.valueOf(ClientMetaData.getWebsiteId()))
                .addParam(StaticFields.API_KEY, ClientMetaData.getApiKey(context))
                .setMethod(Request.Method.POST)
                .setUrl(ApiInfo.RECEIVE_NOTIFICATION.getValue())
                .build();
        WorkManager.getInstance(context).enqueue(request);
    }

    private static String getNotificationType(String type) {
        return type != null ? type : StaticFields.NOTIFICATION_TYPE;
    }

    public static void handleNewToken(@NonNull final Context context) {

        if (!NajvaClient.configuration.isFirebaseEnabled()) {
            FirebaseInstanceId
                    .getInstance(FirebaseApp.getInstance(StaticFields.NAJVA))
                    .getInstanceId()
                    .addOnSuccessListener(new OnSuccessListener<InstanceIdResult>() {
                        @Override
                        public void onSuccess(InstanceIdResult instanceIdResult) {
                            sendTokenToServer(context, instanceIdResult.getToken());
                        }
                    });
        } else {
            FirebaseInstanceId
                    .getInstance()
                    .getInstanceId()
                    .addOnSuccessListener(new OnSuccessListener<InstanceIdResult>() {
                        @Override
                        public void onSuccess(InstanceIdResult instanceIdResult) {
                            sendTokenToServer(context, instanceIdResult.getToken());
                        }
                    });
        }

    }

    private static void sendTokenToServer(Context context, String token) {
        Log.d("PushNotificationHandler", "new token: " + token);
        String lastToken = FileManager.readFromFile(context, Parameter.FIREBASE_TOKEN.getValue());
        if (lastToken != null && lastToken.equals(token)) return;

        ClientMetaData.setToken(token);
        CampaignController campaignController = new CampaignControllerImpl(
                context,
                new LocationControllerImpl(context),
                new RemoteConfigControllerImpl(context),
                ClientMetaData.getParams()
        );
        campaignController.load();
    }

    public static boolean isNajvaMessage(@NonNull Context context, RemoteMessage remoteMessage) {
        return NotificationUtils.getInstance().isNajvaMessage(context, remoteMessage);
    }


}
