package com.najva.sdk.login;

import androidx.lifecycle.LiveData;
import androidx.lifecycle.Observer;
import android.content.Context;
import android.os.Handler;
import android.os.Looper;

import androidx.annotation.Nullable;

import androidx.work.OneTimeWorkRequest;
import androidx.work.WorkInfo;
import androidx.work.WorkManager;

import com.najva.sdk.core.Utility;
import com.najva.sdk.core.utils.ApiInfo;
import com.najva.sdk.core.utils.CookieManager;
import com.najva.sdk.core.utils.FileManager;
import com.najva.sdk.core.utils.Logger;
import com.najva.sdk.core.utils.Parameter;
import com.najva.sdk.core.utils.StaticFields;
import com.najva.sdk.core.works.FormRequestWorker;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Locale;
import java.util.concurrent.TimeUnit;

/**
 * Save last login time and send it to server
 */
public class LoginControllerImpl extends Thread implements LoginController {

    private static String TAG = "LoginController";

    private final Context context;
    private final String apiKey;
    private final int websiteId;
    private final String DATA_PATTERN = "MM/dd/yyyy hh:mm:ss";
    @SuppressWarnings("FieldCanBeLocal")
    private final int DURATION = 24;

    public LoginControllerImpl(Context context, String apiKey, int websiteId) {
        this.context = context;
        this.apiKey = apiKey;
        this.websiteId = websiteId;

    }

    @Override
    public void load() {
        updateLastLogin();
    }

    @Override
    public void loadAsync() {
        start();
    }

    @Override
    public void run() {
        load();
    }

    private void updateLastLogin() {
        Logger.i(TAG, "updateLastLogin: initialized");
        if (!checkLastLoginTimeStatus() || !Utility.isSubscribed(context)) {
            return;
        }

        StringBuilder currentUrl = new StringBuilder(ApiInfo.API_CURRENT_USER.getValue() + "?");
        final HashMap<String, String> lastUpdateTimeParams = getLastUpdateTimeParams();
        for (String key : lastUpdateTimeParams.keySet()) {
            currentUrl.append(key).append("=").append(lastUpdateTimeParams.get(key)).append("&");
        }
        if (lastUpdateTimeParams.size() > 0) {
            currentUrl.deleteCharAt(currentUrl.length() - 1);
        }
        String url = currentUrl.toString();


        OneTimeWorkRequest oneTimeWorkRequest = new FormRequestWorker.Builder(context)
                .setUrl(url)
                .build();
        WorkManager.getInstance(context).enqueue(oneTimeWorkRequest);

        final LiveData<WorkInfo> workInfoLiveData = WorkManager.getInstance()
                .getWorkInfoByIdLiveData(oneTimeWorkRequest.getId());

        new Handler(Looper.getMainLooper()).post(new Runnable() {
            @Override
            public void run() {
                workInfoLiveData.observeForever(new Observer<WorkInfo>() {
                    @Override
                    public void onChanged(@Nullable WorkInfo workInfo) {
                        if (workInfo == null) return;
                        if (workInfo.getState() == WorkInfo.State.SUCCEEDED) {
                            saveLastLoginTime();
                            workInfoLiveData.removeObserver(this);
                        }
                    }
                });
            }
        });
    }

    private boolean checkLastLoginTimeStatus() {
        SimpleDateFormat formatter = new SimpleDateFormat(DATA_PATTERN, Locale.US);
        String value = FileManager.readFromFile(this.context, Parameter.LAST_UPDATE_LOGIN_TIME.getValue());
        if (value != null) {
            try {
                Date lastDate = formatter.parse(value);
                if (new Date(System.currentTimeMillis() - TimeUnit.HOURS.toMillis(DURATION)).before(lastDate) &&
                        lastDate.getDay() == new Date(System.currentTimeMillis()).getDay() &&
                        lastDate.getMonth() == new Date(System.currentTimeMillis()).getMonth() &&
                        lastDate.getYear() == new Date(System.currentTimeMillis()).getYear()) {
                    return false;
                }
            } catch (ParseException e) {
                //e.printStackTrace();
            }

        }

        return true;
    }

    private HashMap<String, String> getLastUpdateTimeParams() {
        HashMap<String, String> params = new HashMap<>();

        params.put(StaticFields.UPDATE_TIME, "1");
        params.put(StaticFields.GET_TOKEN, "1");
        params.put(StaticFields.API_KEY, apiKey);
        params.put(StaticFields.WEBSITE, String.valueOf(websiteId));

        String najvaToken = CookieManager.getInstance(context).getCookies();
        if (najvaToken != null) {
            try {
                najvaToken = najvaToken.substring(12, najvaToken.length() - 1);
                params.put(StaticFields.NAJVA_TOKEN, najvaToken);
            } catch (Exception ignored) {
            }
        }

        return params;
    }

    private void saveLastLoginTime() {
        SimpleDateFormat formatter = new SimpleDateFormat(DATA_PATTERN, Locale.US);
        FileManager.writeToFile(this.context, Parameter.LAST_UPDATE_LOGIN_TIME.getValue(),
                formatter.format(new Date(System.currentTimeMillis())));
    }
}
