package com.najva.sdk.crash_report;

import android.content.Context;

import androidx.work.OneTimeWorkRequest;
import androidx.work.WorkManager;

import com.android.volley.Request;
import com.najva.sdk.core.Controller;
import com.najva.sdk.core.utils.ApiInfo;
import com.najva.sdk.core.utils.FileManager;
import com.najva.sdk.core.works.FormRequestWorker;


public class NajvaCrashReporter implements Controller {

    private static final String SEPARATOR = "#";
    private static final String ITEM_SEPARATOR = ";";
    private static final String KEY_VAL_SEPARATOR = "=";
    private static final String CRASH_FILE_NAME = "crash_log";
    private static final String ERROR_MESSAGE = "error_message";
    private static final String THREAD_NAME = "thread_name";
    private static final String STACK_TRACE = "stack_trace";
    private static final String NAJVA_PACKAGE = "com.najva.sdk";

    private Context context;

    public static NajvaCrashReporter getInstance(Context context) {
        return new NajvaCrashReporter(context);
    }

    @SuppressWarnings("FieldCanBeLocal")
    private Thread.UncaughtExceptionHandler najvaExceptionHandler = new Thread.UncaughtExceptionHandler() {
        @Override
        public void uncaughtException(Thread thread, Throwable throwable) {


            if (isNajvaException(throwable.getCause().getStackTrace())) {

                StringBuilder builder = new StringBuilder();

                String content = NajvaCrashReporter.this.toString(throwable.getCause().getStackTrace());

                builder.append(ERROR_MESSAGE).append(KEY_VAL_SEPARATOR).append(throwable.getMessage());
                builder.append(ITEM_SEPARATOR);
                builder.append(THREAD_NAME).append(KEY_VAL_SEPARATOR).append(thread.getName());
                builder.append(ITEM_SEPARATOR);
                builder.append(STACK_TRACE).append(KEY_VAL_SEPARATOR).append(content);
                builder.append(SEPARATOR);

                FileManager.appendToFile(context, CRASH_FILE_NAME, builder.toString());
            }
            if (prevExceptionHandler != null) {
                prevExceptionHandler.uncaughtException(thread, throwable);
            }
        }
    };

    private boolean isNajvaException(StackTraceElement[] stackTrace) {
        boolean isNajvaException = false;
        for (StackTraceElement element : stackTrace) {
            isNajvaException = isNajvaException || element.getClassName().contains(NAJVA_PACKAGE);
        }
        return isNajvaException;
    }

    private Thread.UncaughtExceptionHandler prevExceptionHandler;

    private NajvaCrashReporter(Context context) {

        this.context = context;

        prevExceptionHandler = Thread.getDefaultUncaughtExceptionHandler();

        Thread.setDefaultUncaughtExceptionHandler(najvaExceptionHandler);
    }

    public String toString(StackTraceElement[] elements) {
        StringBuilder builder = new StringBuilder();
        for (StackTraceElement element : elements) {
            builder.append(element.toString());
            builder.append("\n");
        }
        return builder.toString();
    }

    public void flushSavedLogs() {
        String content = FileManager.readFromFile(context, CRASH_FILE_NAME);
        // check if content is not null
        if (content == null) return;
        // trim content and remove unnecessary spacing and lines
        content = content.trim();
        //check if content is not empty
        if (content.isEmpty()) return;

        String[] items = content.split(SEPARATOR);
        for (String item : items) {
            sendItemToServer(item);
        }

        FileManager.deleteFile(context, CRASH_FILE_NAME);
    }

    private void sendItemToServer(String item) {
        if (item.isEmpty()) return;
        FormRequestWorker.Builder requestBuilder = new FormRequestWorker.Builder(context);

        String[] lines = item.split(ITEM_SEPARATOR);

        for (String line : lines) {
            line = line.trim();
            if (!line.isEmpty()) {
                String[] keyVal = line.split(KEY_VAL_SEPARATOR);
                requestBuilder.addParam(keyVal[0], keyVal[1]);
            }
        }

        requestBuilder.setMethod(Request.Method.POST)
                .setUrl(getUrl());

        OneTimeWorkRequest oneTimeWorkRequest = requestBuilder.build();

        WorkManager.getInstance().enqueue(oneTimeWorkRequest);
    }

    private String getUrl() {
        return ApiInfo.CRASH_URL.getValue();
    }


    @Override
    public void load() {
        flushSavedLogs();
    }

    @Override
    public void loadAsync() {
        Runnable runnable = new Runnable() {
            @Override
            public void run() {
                flushSavedLogs();
            }
        };

        new Thread(runnable).start();
    }
}
