package com.najva.sdk.core.works;

import android.content.Context;
import android.location.Location;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import androidx.work.Constraints;
import androidx.work.NetworkType;
import androidx.work.PeriodicWorkRequest;
import androidx.work.WorkerParameters;

import com.android.volley.Request;
import com.android.volley.VolleyError;
import com.najva.sdk.core.utils.ApiInfo;
import com.najva.sdk.core.utils.CookieManager;
import com.najva.sdk.core.utils.IBuilder;

import com.najva.sdk.core.utils.MetaJsonRequest;
import com.najva.sdk.core.utils.StaticFields;
import com.najva.sdk.core.utils.VolleyErrorHandler;
import com.najva.sdk.core.utils.VolleySingleton;
import com.najva.sdk.core.utils.WorkManager;
import com.najva.sdk.location.LocationManagerApiWrapper;



import java.util.HashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;


public class LocationRequestWorker extends BaseWorker {
    private static final String TAG = "LocationRequestWorker";
    private Location location;


    public LocationRequestWorker(@NonNull Context context, @NonNull WorkerParameters workerParams) {
        super(context, workerParams);
    }


    @NonNull
    @Override
    public Result doWork() {

        new Handler(Looper.getMainLooper()).post(
                new Runnable() {
                    @Override
                    public void run() {
                        location = getLocation();

                    }
                }
        );


        try {
            synchronized (this) {
                LocationRequestWorker.this.wait(10000);
            }
        } catch (InterruptedException e) {
            e.printStackTrace();
        }


        if (location != null) {


            //fetch location
            HashMap<String, String> parameter = fetchLocationParameter(location);

            //send location to server
            MetaJsonRequest locationRequest = new MetaJsonRequest.Builder()
                    .setAsync(false)
                    .setUrl(ApiInfo.SET_LOCATION_URL.getValue())
                    .setParams(parameter)
                    .setMethod(Request.Method.POST)
                    .setCookieManager(CookieManager.getInstance(getApplicationContext()))
                    .build();
            VolleySingleton.getInstance(getApplicationContext()).add(locationRequest);

            try {
                locationRequest.getFuture().get();
            }  // Interrupt exception is for threading problems
            catch (InterruptedException e) {
                Log.d(TAG, "doWork: InterruptedException");
                return Result.retry();
            }
            // error is now for volley
            catch (ExecutionException e) {
                Log.d(TAG, "doWork: ExecutionException");

                if (e.getCause() instanceof VolleyError) {
                    Result result = VolleyErrorHandler.handleError((VolleyError) e.getCause());
                    if (result.equals(Result.failure())) {
                        locationRequest.getFuture().onErrorResponse((VolleyError) e.getCause());
                    }
                    return result;
                }
                return Result.retry();
            }
            // timeout error
            catch (AssertionError e) {
                Log.d(TAG, "doWork: AssertionError ");
                return Result.retry();
            }

            Log.d(TAG, "doWork: succeed " + parameter.toString());

            return Result.success();
        } else {
            Log.d(TAG, "doWork: failure");
            return Result.failure();
        }
    }

    @Nullable
    private Location getLocation() {
        try {
            return LocationManagerApiWrapper.getInstance(getApplicationContext()).getLocation(getApplicationContext());
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    @NonNull
    private HashMap<String, String> fetchLocationParameter(@NonNull Location location) {
        HashMap<String, String> map = new HashMap<>();
        map.put(StaticFields.ServerParameter.LATITUDE, String.valueOf(location.getLatitude()));
        map.put(StaticFields.ServerParameter.LONGITUDE, String.valueOf(location.getLongitude()));
        map.put(StaticFields.ServerParameter.ALTITUDE, String.valueOf(location.getAltitude()));
        map.put(StaticFields.ServerParameter.ACCURACY, String.valueOf(location.getAccuracy()));
        return map;
    }

    public static class Builder implements IBuilder<PeriodicWorkRequest> {
        private long repeatInterval = StaticFields.DefaultParameter.REPEAT_INTERVAL;

        public Builder() {
        }

        public Builder setRepeatInterval(long repeatInterval) {
            this.repeatInterval = repeatInterval;
            return this;
        }

        @NonNull
        @Override

        public PeriodicWorkRequest build() {
            Constraints constraints = new Constraints.Builder()
                    .setRequiredNetworkType(NetworkType.CONNECTED)
                    .build();
            return new PeriodicWorkRequest.Builder(LocationRequestWorker.class, repeatInterval, TimeUnit.MINUTES)
                    .setConstraints(constraints)
                    .addTag(WorkManager.TAG)
                    .addTag(TAG)
                    .build();
        }
    }
}
