package com.najva.sdk.core.works;

import android.content.Context;
import android.content.SharedPreferences;
import android.support.annotation.NonNull;
import android.util.Log;

import androidx.work.BackoffPolicy;
import androidx.work.Constraints;
import androidx.work.Data;
import androidx.work.NetworkType;
import androidx.work.OneTimeWorkRequest;
import androidx.work.WorkerParameters;

import com.android.volley.Request;
import com.android.volley.VolleyError;
import com.najva.sdk.core.utils.CookieManager;
import com.najva.sdk.core.utils.IBuilder;
import com.najva.sdk.core.utils.MetaFormRequest;
import com.najva.sdk.core.utils.VolleyErrorHandler;
import com.najva.sdk.core.utils.StaticFields;
import com.najva.sdk.core.utils.VolleySingleton;
import com.najva.sdk.core.utils.WorkManager;

import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

public class FormRequestWorker extends BaseWorker {
    private static final String TAG = "FormRequestWorker";
    private MetaFormRequest request;
    private String name;

    public FormRequestWorker(@NonNull Context context, @NonNull WorkerParameters workerParams) {
        super(context, workerParams);
        //initiateRequest();
    }

    private void initiateRequest() {
        name = getInputData().getString(StaticFields.RequestParameter.NAME);
        this.request = new MetaFormRequest.Builder()
                .setAsync(false)
                .setCookieManager(CookieManager.getInstance(getApplicationContext()))
                .setUrl(getUrl())
                .setParams(getPostParams())
                .setHeaders(getHeaders())
                .setMethod(getMethod())
                .build();
        Log.d(TAG, "params: " + getPostParams().toString());
        Log.d(TAG, "header: " + getHeaders().toString());
        Log.d(TAG, "method: " + getMethod());
    }

    private Map<String, String> getPostParams() {
        Map<String, String> params = new HashMap<>();
        SharedPreferences workerMap = getApplicationContext().getSharedPreferences(StaticFields.RequestParameter.POST + name, Context.MODE_PRIVATE);
        for (String key : workerMap.getAll().keySet()) {
            params.put(key, workerMap.getString(key, ""));
        }
        return params;
    }

    private Map<String, String> getHeaders() {
        Map<String, String> headers = new HashMap<>();
        SharedPreferences workerMap = getApplicationContext().getSharedPreferences(StaticFields.RequestParameter.HEADER + name, Context.MODE_PRIVATE);

        for (String key : workerMap.getAll().keySet()) {
            headers.put(key, workerMap.getString(key, ""));
        }
        return headers;
    }

    private int getMethod() {
        return getInputData().getInt(StaticFields.RequestParameter.METHOD, Request.Method.GET);
    }

    private String getUrl() {
        return getInputData().getString(StaticFields.RequestParameter.URL);
    }

    @NonNull
    @Override
    public Result doWork() {
        Log.d(TAG,"sending request");
        initiateRequest();
        try {
            VolleySingleton.getInstance(getApplicationContext()).add(request);
            String response = request.getFuture().get();

            request.getFuture().onResponse(response);
            getApplicationContext().getSharedPreferences("post" + name, Context.MODE_PRIVATE).edit().clear().apply();
            getApplicationContext().getSharedPreferences("header" + name, Context.MODE_PRIVATE).edit().clear().apply();
            return Result.success(new Data.Builder().putString("response", response).build());
        }
        // Interrupt exception is for threading problems
        catch (InterruptedException e) {
            return Result.retry();
        }
        // error is now for volley
        catch (ExecutionException e) {
            if (e.getCause() instanceof VolleyError) {
                Result result = VolleyErrorHandler.handleError((VolleyError) e.getCause());
                if (result.equals(Result.failure())){
                    request.getFuture().onErrorResponse((VolleyError) e.getCause());
                }
                return result;
            }
            return Result.retry();
        }
        // timeout error
        catch (AssertionError e){
            return Result.retry();
        }
    }

    public static class Builder implements IBuilder<OneTimeWorkRequest> {
        private String url;
        private int method = Request.Method.GET;
        private SharedPreferences params;
        private SharedPreferences headers;
        private String name;

        public Builder(@NonNull Context context) {
            // random name
            // when using a single sharedPref for requests it may cause problems like
            // when you have two or more workers at same time and this causes conflicts
            // to the params and headers.
            name = generateRandom();

            params = context.getSharedPreferences(StaticFields.RequestParameter.POST + name, Context.MODE_PRIVATE);
            headers = context.getSharedPreferences(StaticFields.RequestParameter.HEADER + name, Context.MODE_PRIVATE);
        }

        @NonNull
        private String generateRandom() {
            return UUID.randomUUID().toString().replaceAll("-", "");
        }

        public Builder setUrl(@NonNull String url) {
            this.url = url;
            return this;
        }

        public Builder setMethod(int method) {
            this.method = method;
            return this;
        }

        public Builder addParam(@NonNull String key,@NonNull String value) {
            params.edit().putString(key, value).apply();
            return this;
        }

        public Builder addHeader(@NonNull String key,@NonNull String value) {
            headers.edit().putString(key, value).apply();
            return this;
        }

        public Builder setParams(@NonNull Map<String, String> params) {
            SharedPreferences.Editor editor = this.params.edit();
            for (String key : params.keySet()) {
                editor.putString(key, params.get(key));
            }
            editor.apply();
            return this;
        }

        public Builder setHeaders(@NonNull Map<String, String> headers) {
            SharedPreferences.Editor editor = this.headers.edit();
            for (String key : headers.keySet()) {
                editor.putString(key, headers.get(key));
            }
            editor.apply();
            return this;
        }

        @NonNull
        @Override
        public OneTimeWorkRequest build() {
            Constraints constraints = new Constraints.Builder()
                    .setRequiredNetworkType(NetworkType.CONNECTED)
                    .build();

            Data.Builder dataBuilder = new Data.Builder();

            dataBuilder.putString(StaticFields.RequestParameter.URL, url);
            dataBuilder.putInt(StaticFields.RequestParameter.METHOD, method);
            dataBuilder.putString(StaticFields.RequestParameter.NAME, name);

            Log.d(TAG, "max allowed size: " + Data.MAX_DATA_BYTES);

            return new OneTimeWorkRequest.Builder(FormRequestWorker.class)
                    .setInputData(dataBuilder.build()).setConstraints(constraints)
                    .setBackoffCriteria(BackoffPolicy.EXPONENTIAL, 1, TimeUnit.MINUTES)
                    .addTag(WorkManager.TAG)
                    .addTag(TAG)
                    .build();
        }
    }

}
