package com.najva.najvasdk.Utils;

import android.content.Context;
import android.location.Location;
import android.location.LocationListener;
import android.location.LocationManager;
import android.os.Bundle;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.google.android.gms.common.internal.Preconditions;

import java.math.BigDecimal;

import static android.Manifest.permission.ACCESS_COARSE_LOCATION;
import static android.Manifest.permission.ACCESS_FINE_LOCATION;

/**
 * A helper class for using android location api.
 * This class use GPS and Network Location
 * These permission is mandatory for using this class
 * - GPS mandatory permissions : ACCESS_FINE_LOCATION
 * - NETWORK_LOCATION mandatory permissions: ACCESS_FINE_LOCATION and ACCESS_COARSE_LOCATION
 *
 * For getting device location use {@link #getLastKnownLocation(Context, int, LocationAwareness)}
 */
public class LocationManagerApiWrapper implements LocationListener {

    private static volatile LocationManagerApiWrapper instance;

    private Location location;

    public static LocationManagerApiWrapper getInstance(Context context) {
        if (instance == null) {
            synchronized (LocationManagerApiWrapper.class) {
                if (instance == null) {
                    instance = new LocationManagerApiWrapper(context);
                }
            }
        }
        return instance;
    }

    private LocationManagerApiWrapper(Context context) {
        this.location = getLastKnownLocation(context, 0, LocationAwareness.NORMAL);

        final LocationManager locationManager =
                (LocationManager) context.getSystemService(Context.LOCATION_SERVICE);


        if (ValidLocationProvider.GPS.hasRequiredPermissions(context)) {
            locationManager.requestLocationUpdates(
                    LocationManager.GPS_PROVIDER,
                    1000,
                    100,
                    this
            );
        }
        else if (ValidLocationProvider.NETWORK.hasRequiredPermissions(context)) {
            locationManager.requestLocationUpdates(
                    LocationManager.NETWORK_PROVIDER,
                    1000,
                    100,
                    this
            );
        }
        else {
            NajvaLog.debug("GPS or Network permission does not granted");
        }
    }

    public Location getTruncatedLocation(Context context) {
        if (ValidLocationProvider.GPS.hasRequiredPermissions(context) || ValidLocationProvider.NETWORK.hasRequiredPermissions(context)) {
            truncateLocationLatLon(location, 8);
            return location;
        }
        else {
            NajvaLog.debug("GPS or Network permission does not granted");
            return null;
        }
    }

    public Location getLocation(Context context) {
        if (ValidLocationProvider.GPS.hasRequiredPermissions(context) || ValidLocationProvider.NETWORK.hasRequiredPermissions(context)) {
            return location;
        }
        else {
            NajvaLog.debug("GPS or Network permission does not granted");
            return null;
        }
    }

    @Override
    public void onLocationChanged(Location location) {
        this.location = location;
    }

    @Override
    public void onStatusChanged(String provider, int status, Bundle extras) {
        NajvaLog.debug("Location status changed, " + status);
    }

    @Override
    public void onProviderEnabled(String provider) {
        NajvaLog.debug("Location provider enabled");
    }

    @Override
    public void onProviderDisabled(String provider) {
        NajvaLog.debug("Location provide disabled");

    }

    public enum ValidLocationProvider {
        NETWORK(LocationManager.NETWORK_PROVIDER),
        GPS(LocationManager.GPS_PROVIDER);

        @NonNull
        final String name;

        ValidLocationProvider(@NonNull final String name) {
            this.name = name;
        }

        @Override
        public String toString() {
            return name;
        }

        private boolean hasRequiredPermissions(@NonNull final Context context) {
            switch (this) {
                case NETWORK:
                    return DeviceUtils.isPermissionGranted(context, ACCESS_FINE_LOCATION)
                            || DeviceUtils.isPermissionGranted(context, ACCESS_COARSE_LOCATION);
                case GPS:
                    return DeviceUtils.isPermissionGranted(context, ACCESS_FINE_LOCATION);
                default:
                    return false;
            }
        }
    }

    /**
     * Returns the last known location of the device using its GPS and network location providers.
     * May be null if:
     * - Location permissions are not requested in the Android manifest file
     * - The location providers don't exist
     *
     * @return instance of {@link Location} that contains longitude and latitude
     */
    @Nullable
    private static Location getLastKnownLocation(@NonNull final Context context,
                                                final int locationPrecision,
                                                final @NonNull LocationAwareness locationAwareness) {
        Preconditions.checkNotNull(context);
        Preconditions.checkNotNull(locationAwareness);

        if (locationAwareness == LocationAwareness.DISABLED) {
            return null;
        }

        final Location gpsLocation = getLocationFromProvider(context,
                ValidLocationProvider.GPS);
        final Location networkLocation = getLocationFromProvider(context,
                ValidLocationProvider.NETWORK);

        if (gpsLocation == null && networkLocation == null) {
            return null;
        }

        final Location result = getMostRecentValidLocation(gpsLocation, networkLocation);


        // Truncate latitude/longitude to the number of digits specified by locationPrecision.
        if (locationAwareness == LocationAwareness.TRUNCATED) {
            truncateLocationLatLon(result, locationPrecision);
        }

        return result;
    }

    @Nullable
    private static Location getLocationFromProvider(@NonNull final Context context,
                                            @NonNull final ValidLocationProvider provider) {

        Location location = null;

        Preconditions.checkNotNull(context);
        Preconditions.checkNotNull(provider);

        if (provider.hasRequiredPermissions(context)) {
            final LocationManager locationManager =
                    (LocationManager) context.getSystemService(Context.LOCATION_SERVICE);
            try {
                // noinspection ResourceType
                location = locationManager.getLastKnownLocation(provider.toString());
            } catch (SecurityException e) {
            NajvaLog.debug("Failed to retrieve location from " +
                    provider.toString() + " provider: access appears to be disabled.");
            } catch (IllegalArgumentException | NullPointerException e) {
            NajvaLog.debug("Failed to retrieve location: device has no " +
                    provider.toString() + " location provider.");
            } catch (Exception e) {
                NajvaLog.debug("Unknown error occurred when getLocationFromProvider, " + (e.getMessage() != null ? e.getMessage() : ""));
            }

        }
        else {
            NajvaLog.debug("GPS or Network permission does not granted");
        }

        return location;
    }

    @Nullable
    private static Location getMostRecentValidLocation(@Nullable final Location a,
                                                       @Nullable final Location b) {
        if (a == null) {
            return b;
        }

        if (b == null) {
            return a;
        }

        // At this point, locations A and B are non-null, so return the more recent one
        return (a.getTime() > b.getTime()) ? a : b;
    }

    private static void truncateLocationLatLon(@Nullable final Location location,
            final int precision) {
        if (location == null || precision < 0) {
            return;
        }

        double lat = location.getLatitude();
        double truncatedLat = BigDecimal.valueOf(lat)
                .setScale(precision, BigDecimal.ROUND_HALF_DOWN)
                .doubleValue();
        location.setLatitude(truncatedLat);

        double lon = location.getLongitude();
        double truncatedLon = BigDecimal.valueOf(lon)
                .setScale(precision, BigDecimal.ROUND_HALF_DOWN)
                .doubleValue();
        location.setLongitude(truncatedLon);
    }
}
