package com.mysoftsource.rxandroidauth2.impl;

import android.app.Activity;
import android.content.Intent;
import android.support.annotation.NonNull;

import com.google.android.gms.auth.api.signin.GoogleSignIn;
import com.google.android.gms.auth.api.signin.GoogleSignInAccount;
import com.google.android.gms.auth.api.signin.GoogleSignInClient;
import com.google.android.gms.auth.api.signin.GoogleSignInOptions;
import com.google.android.gms.common.api.ApiException;
import com.google.android.gms.tasks.OnCompleteListener;
import com.google.android.gms.tasks.Task;
import com.google.firebase.auth.AuthCredential;
import com.google.firebase.auth.GoogleAuthProvider;
import com.mysoftsource.rxandroidauth2.R;
import com.mysoftsource.rxandroidauth2.exception.GoogleAuthException;
import com.mysoftsource.rxandroidauth2.util.Output;
import com.mysoftsource.rxandroidauth2.util.RxLog;

import io.reactivex.Observable;
import io.reactivex.ObservableEmitter;
import io.reactivex.ObservableOnSubscribe;
import io.reactivex.disposables.Disposable;
import io.reactivex.functions.Action;
import io.reactivex.functions.Cancellable;
import io.reactivex.functions.Consumer;
import io.reactivex.functions.Function;

public final class GoogleAuthSocialNetwork extends AuthSocialNetwork {
    private static final int RC_SIGN_IN = 1000;

    private GoogleSignInClient mGoogleSignInClient;
    private final Output<String> mFirebaseIdTokenOutputRelay;
    private final Function<Throwable, Observable<?>> errorMapper = new Function<Throwable, Observable<?>>() {
        @Override
        public Observable<?> apply(Throwable throwable) throws Exception {
            return Observable.error(throwable);
        }
    };


    public GoogleAuthSocialNetwork(Activity activity) {
        super(activity);
        GoogleSignInOptions gso = new GoogleSignInOptions.Builder(GoogleSignInOptions.DEFAULT_SIGN_IN)
                .requestIdToken(activity.getString(R.string.default_web_client_id))
                .requestEmail()
                .build();

        mGoogleSignInClient = GoogleSignIn.getClient(activity, gso);
        mFirebaseIdTokenOutputRelay = new Output<>();
    }

    @Override
    protected Observable<String> requestLogin() {
        return Observable.create(new ObservableOnSubscribe<String>() {
            @Override
            public void subscribe(ObservableEmitter<String> emitter) throws Exception {
                Disposable disposable = withErrorHandling(mFirebaseIdTokenOutputRelay)
                        .take(1)
                        .subscribe(new Consumer<String>() {
                                       @Override
                                       public void accept(String s) throws Exception {
                                           emitter.onNext(s);
                                       }
                                   },
                                new Consumer<Throwable>() {
                                    @Override
                                    public void accept(Throwable throwable) throws Exception {
                                        emitter.onError(throwable);
                                    }
                                },
                                new Action() {
                                    @Override
                                    public void run() throws Exception {
                                        emitter.onComplete();
                                    }
                                }
                        );

                Intent signInIntent = mGoogleSignInClient.getSignInIntent();
                mActivity.startActivityForResult(signInIntent, RC_SIGN_IN);
                emitter.setCancellable(new Cancellable() {
                    @Override
                    public void cancel() throws Exception {
                        disposable.dispose();
                    }
                });
            }
        });
    }

    @Override
    protected Observable<Boolean> requestLogout() {
        return Observable.create(emitter -> {
            mGoogleSignInClient.signOut().addOnCompleteListener(new OnCompleteListener<Void>() {
                @Override
                public void onComplete(@NonNull Task<Void> task) {
                    RxLog.d("requestLogout>> google is signOut");
                    emitter.onNext(task.isSuccessful());
                    emitter.onComplete();
                }
            });
        });
    }

    @Override
    protected boolean isLoggedIn() {
        return GoogleSignIn.getLastSignedInAccount(mActivity) != null;
    }

    @Override
    public void onActivityResult(int requestCode, int resultCode, Intent data) {
        super.onActivityResult(requestCode, resultCode, data);
        if (requestCode == RC_SIGN_IN && mFirebaseIdTokenOutputRelay.hasObservers()) {
            if (resultCode == Activity.RESULT_CANCELED) {
                mFirebaseIdTokenOutputRelay.errorRelay.accept(new GoogleAuthException("Google sign in cancel", GoogleAuthException.Reason.CANCEL));
                return;
            }
            Task<GoogleSignInAccount> task = GoogleSignIn.getSignedInAccountFromIntent(data);
            try {
                // Google Sign In was successful, authenticate with Firebase
                RxLog.d("Google Sign In was successful, authenticate with Firebase");
                GoogleSignInAccount account = task.getResult(ApiException.class);
                firebaseAuthWithGoogle(account)
                        .subscribe(idToken -> mFirebaseIdTokenOutputRelay.valueRelay.accept(idToken),
                                throwable -> mFirebaseIdTokenOutputRelay.errorRelay.accept(new GoogleAuthException("firebase auth failed", throwable)));
            } catch (ApiException e) {
                // Google Sign In failed, update UI appropriately
                RxLog.e(e, "Google sign in failed");
                mFirebaseIdTokenOutputRelay.errorRelay.accept(new GoogleAuthException("Google sign in failed", e));
            }
        }
    }

    private Observable<String> firebaseAuthWithGoogle(GoogleSignInAccount account) {
        return Observable.create(new ObservableOnSubscribe<String>() {
            @Override
            public void subscribe(ObservableEmitter<String> emitter) throws Exception {
                RxLog.d("firebaseAuthWithGoogle>> token = %s", account.getIdToken());
                AuthCredential credential = GoogleAuthProvider.getCredential(account.getIdToken(), null);
                handleSocialNetworkAccessToken(credential, emitter);
            }
        });
    }

    private <T> Observable<T> withErrorHandling(Output<T> output) {
        return Observable.merge(
                output.valueRelay,
                (Observable<T>) output.errorRelay.flatMap(errorMapper)
        );
    }
}
