package com.mysoftsource.rxandroidauth2;

import android.app.Activity;
import android.content.Intent;

import com.mysoftsource.rxandroidauth2.exception.RxAuthException;
import com.mysoftsource.rxandroidauth2.impl.FacebookAuthSocialNetwork;
import com.mysoftsource.rxandroidauth2.impl.AuthSocialNetwork;
import com.mysoftsource.rxandroidauth2.impl.GoogleAuthSocialNetwork;
import com.mysoftsource.rxandroidauth2.util.RxLog;

import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

import io.reactivex.Observable;

public class AuthHelper {
    public enum SocialNetworkType {
        FACEBOOK,
        GOOGLE
    }

    private Map<SocialNetworkType, AuthSocialNetwork> SOCIAL_NETWORKS;

    public AuthHelper(Map<SocialNetworkType, AuthSocialNetwork> socialNetworkMap) {
        SOCIAL_NETWORKS = socialNetworkMap;
    }

    public Observable<String> login(SocialNetworkType type) {
        if (!SOCIAL_NETWORKS.containsKey(type)) {
            throw new RxAuthException(String.format(Locale.US, "%s wasn't initialized... in AuthHelper.Builder", type.toString()));
        }
        if (isNotAuthAvailable()) {
            return Observable.error(new RxAuthException("contain auth is busy, please check logic"));
        }
        return SOCIAL_NETWORKS.get(type).login();
    }

    public Observable<Boolean> logout(SocialNetworkType type) {
        if (!SOCIAL_NETWORKS.containsKey(type)) {
            throw new RxAuthException(String.format(Locale.US, "%s wasn't initialized... in AuthHelper.Builder", type.toString()));
        }
        if (isNotAuthAvailable()) {
            return Observable.error(new RxAuthException("contain auth is busy, please check logic"));
        }
        return SOCIAL_NETWORKS.get(type).logout();
    }

    public Observable<Boolean> logout() {
        if (SOCIAL_NETWORKS.isEmpty()) {
            throw new RxAuthException("no function initialized... in AuthHelper.Builder");
        }
        if (isNotAuthAvailable()) {
            return Observable.error(new RxAuthException("contain auth is busy, please check logic"));
        }
        return Observable.fromIterable(SOCIAL_NETWORKS.values())
                .concatMap(authSocialNetwork -> authSocialNetwork.logout())
                .doOnNext(success -> RxLog.d("logout is success %b", success));
    }

    public void onActivityResult(int requestCode, int resultCode, Intent data) {
        for (AuthSocialNetwork authSocialNetwork : SOCIAL_NETWORKS.values()) {
            authSocialNetwork.onActivityResult(requestCode, resultCode, data);
        }
    }

    private boolean isNotAuthAvailable() {
        boolean isAuthBusy = false;
        for (Map.Entry<SocialNetworkType, AuthSocialNetwork> entry : SOCIAL_NETWORKS.entrySet()) {
            if (entry.getValue().isAuthBusy()) {
                RxLog.d("isNotAuthAvailable>> %s is busy", entry.getKey().toString());
                isAuthBusy = true;
            }
        }
        return isAuthBusy;
    }

    public static class Builder {
        private Map<SocialNetworkType, AuthSocialNetwork> socialNetworkMap = new HashMap<>();
        private Activity activity;

        public Builder from(Activity activity) {
            this.activity = activity;
            return this;
        }

        public Builder facebook() {
            checkValidInput();
            if (socialNetworkMap.containsKey(SocialNetworkType.FACEBOOK)) {
                throw new RxAuthException("FACEBOOK type already logged");
            }
            socialNetworkMap.put(SocialNetworkType.FACEBOOK, new FacebookAuthSocialNetwork(activity));
            return this;
        }

        public Builder google(){
            checkValidInput();
            if (socialNetworkMap.containsKey(SocialNetworkType.GOOGLE)) {
                throw new RxAuthException("GOOGLE type already logged");
            }
            socialNetworkMap.put(SocialNetworkType.GOOGLE, new GoogleAuthSocialNetwork(activity));
            return this;
        }

        public AuthHelper build() {
            if (socialNetworkMap.isEmpty()) {
                throw new RxAuthException("no social network is chosen");
            }
            AuthHelper authHelper = new AuthHelper(socialNetworkMap);
            return authHelper;
        }

        private void checkValidInput() {
            if (activity == null) {
                throw new RxAuthException("Please call from() method before all");
            }
        }
    }
}
