package com.mysoftsource.rxandroidauth2.impl;

import android.app.Activity;
import android.content.Intent;

import com.facebook.AccessToken;
import com.facebook.CallbackManager;
import com.facebook.FacebookCallback;
import com.facebook.FacebookException;
import com.facebook.login.LoginManager;
import com.facebook.login.LoginResult;
import com.google.firebase.auth.AuthCredential;
import com.google.firebase.auth.FacebookAuthProvider;
import com.mysoftsource.rxandroidauth2.exception.FacebookAuthException;
import com.mysoftsource.rxandroidauth2.util.RxLog;

import java.util.Arrays;

import io.reactivex.Observable;
import io.reactivex.ObservableEmitter;
import io.reactivex.ObservableOnSubscribe;
import io.reactivex.functions.Cancellable;

public final class FacebookAuthSocialNetwork extends AuthSocialNetwork {
    private CallbackManager mCallbackManager;

    public FacebookAuthSocialNetwork(Activity activity) {
        super(activity);
        mCallbackManager = CallbackManager.Factory.create();
    }

    @Override
    public void onActivityResult(int requestCode, int resultCode, Intent data) {
        super.onActivityResult(requestCode, resultCode, data);
        mCallbackManager.onActivityResult(requestCode, resultCode, data);
    }

    @Override
    protected Observable<String> requestLogin() {
        return Observable.create(new ObservableOnSubscribe<String>() {
            @Override
            public void subscribe(ObservableEmitter<String> emitter) throws Exception {
                LoginManager.getInstance().registerCallback(mCallbackManager, new FacebookCallback<LoginResult>() {
                    @Override
                    public void onSuccess(LoginResult loginResult) {
                        RxLog.d("facebook:onSuccess: %s", loginResult.getAccessToken().getToken());
                        AuthCredential credential = FacebookAuthProvider.getCredential(loginResult.getAccessToken().getToken());
                        handleSocialNetworkAccessToken(credential, emitter);
                    }

                    @Override
                    public void onCancel() {
                        RxLog.d("facebook:onCancel");
                        emitter.onError(new FacebookAuthException("facebook:onCancel", FacebookAuthException.Reason.CANCEL));
                    }

                    @Override
                    public void onError(FacebookException error) {
                        RxLog.e(error, "facebook:onError");
                        emitter.onError(new FacebookAuthException("facebook:onError", error));
                    }
                });
                LoginManager.getInstance().logInWithReadPermissions(mActivity, Arrays.asList("email", "public_profile"));
                emitter.setCancellable(new Cancellable() {
                    @Override
                    public void cancel() throws Exception {
                        LoginManager.getInstance().unregisterCallback(mCallbackManager);
                    }
                });
            }
        });
    }

    @Override
    protected Observable<Boolean> requestLogout() {
        LoginManager.getInstance().logOut();
        return Observable.just(true);
    }

    @Override
    protected boolean isLoggedIn() {
        return AccessToken.getCurrentAccessToken() != null;
    }
}
