package com.mysoftsource.rxandroidauth2.impl;

import android.app.Activity;
import android.content.Intent;
import android.support.annotation.NonNull;

import com.google.android.gms.tasks.OnCompleteListener;
import com.google.android.gms.tasks.Task;
import com.google.firebase.auth.AuthCredential;
import com.google.firebase.auth.AuthResult;
import com.google.firebase.auth.FirebaseAuth;
import com.google.firebase.auth.FirebaseUser;
import com.google.firebase.auth.GetTokenResult;
import com.mysoftsource.rxandroidauth2.exception.RxAuthException;
import com.mysoftsource.rxandroidauth2.util.RxLog;

import io.reactivex.Emitter;
import io.reactivex.Observable;
import io.reactivex.disposables.Disposable;
import io.reactivex.functions.Consumer;

public abstract class AuthSocialNetwork {
    protected FirebaseAuth mFirebaseAuthAuth;
    protected Activity mActivity;
    private boolean mIsLoginBusy;
    private boolean mIsLogoutBusy;

    AuthSocialNetwork(Activity activity) {
        mFirebaseAuthAuth = FirebaseAuth.getInstance();
        mActivity = activity;
        mIsLoginBusy = false;
        mIsLogoutBusy = false;
    }

    public void onActivityResult(int requestCode, int resultCode, Intent data) {
        // override if using
    }

    public Observable<String> login() {
        return requestLogoutIfNeed()
                .flatMap(success -> {
                    RxLog.d("login>> request login");
                    return requestLogin();
                })
                .doOnSubscribe(new Consumer<Disposable>() {
                    @Override
                    public void accept(Disposable disposable) throws Exception {
                        RxLog.d("login>> doOnSubscribe");
                        mIsLoginBusy = true;
                    }
                })
                .doFinally(() -> {
                    RxLog.d("login>> doFinally");
                    mIsLoginBusy = false;
                });
    }

    public Observable<Boolean> logout() {
        RxLog.d("logout>> request logout");
        return requestLogout()
                .doOnNext(success -> mFirebaseAuthAuth.signOut())
                .doOnSubscribe(new Consumer<Disposable>() {
                    @Override
                    public void accept(Disposable disposable) throws Exception {
                        RxLog.d("logout>> doOnSubscribe");
                        mIsLoginBusy = true;
                    }
                })
                .doFinally(() -> {
                    RxLog.d("logout>> doFinally");
                    mIsLoginBusy = false;
                });
    }

    public boolean isAuthBusy() {
        return mIsLoginBusy || mIsLogoutBusy;
    }

    protected abstract Observable<String> requestLogin();

    protected abstract Observable<Boolean> requestLogout();

    protected abstract boolean isLoggedIn();

    protected void handleSocialNetworkAccessToken(AuthCredential credential, final Emitter<String> emitter) {
        RxLog.d("handleSocialNetworkAccessToken>> start signInWithCredential");

        mFirebaseAuthAuth.signInWithCredential(credential)
                .addOnCompleteListener(new OnCompleteListener<AuthResult>() {
                    @Override
                    public void onComplete(@NonNull Task<AuthResult> task) {
                        if (task.isSuccessful()) {
                            // Sign in success, update UI with the signed-in user's information
                            RxLog.d("signInWithCredential:success");
                            FirebaseUser user = mFirebaseAuthAuth.getCurrentUser();
                            handleFirebaseIdToken(user, emitter);
                        } else {
                            // If sign in fails, display a message to the user.
                            RxLog.e(task.getException(), "signInWithCredential:failure");
                            emitter.onError(new RxAuthException(task.getException()));
                        }
                    }
                });
    }

    private void handleFirebaseIdToken(FirebaseUser user, final Emitter<String> emitter) {
        RxLog.d("handleFirebaseAccessToken>> email = %s", user.getEmail());
        RxLog.d("handleFirebaseAccessToken>> Uid = %s", user.getUid());
        user.getIdToken(true).addOnCompleteListener(new OnCompleteListener<GetTokenResult>() {
            @Override
            public void onComplete(@NonNull Task<GetTokenResult> task) {
                if (task.isSuccessful()) {
                    String idToken = task.getResult().getToken();
                    RxLog.d("handleFirebaseIdToken>> idToken = %s", idToken);
                    emitter.onNext(idToken);
                    emitter.onComplete();
                } else {
                    RxLog.e(task.getException(), "handleFirebaseIdToken>> get idToken is failed");
                    emitter.onError(new RxAuthException(task.getException()));
                }
            }
        });
    }

    private Observable<Boolean> requestLogoutIfNeed() {
        if (isLoggedIn()) {
            RxLog.d("Already logged, please check isLoggedIn() method");
            return requestLogout()
                    .doOnNext(success -> mFirebaseAuthAuth.signOut());
        }
        return Observable.just(true);
    }
}
