/*
 * Copyright (c) 2009 Mysema Ltd.
 * All rights reserved.
 * 
 */
package com.mysema.query.types.expr;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.mysema.commons.lang.Assert;
import com.mysema.query.types.Visitor;




/**
 * EStringConst represents String constants
 * 
 * @author tiwe
 *
 */
@SuppressWarnings("serial")
public class EStringConst extends EString implements Constant<String>{
    
    private static final Map<String,EString> cache;
    
    static{
        List<String> strs = new ArrayList<String>(Arrays.asList("", ".", ".*", "%"));
        for (int i = 0; i < 256; i++){
            strs.add(String.valueOf(i));
        }
    
        cache = new HashMap<String,EString>(strs.size());
        for (String str : strs){
            cache.put(str, new EStringConst(str));
        }
    }
    
    
    /**
     * Factory method for constants
     * 
     * @param str
     * @return
     */
    public static final EString create(String str){
        return create(str, false);
    }
    
    public static EString create(String str, boolean populateCache) {
        if (cache.containsKey(str)){
            return cache.get(str);            
        }else{
            EString rv = new EStringConst(Assert.notNull(str));
            if (populateCache) cache.put(str, rv);                
            return rv;
        }
    }
    
    private final String constant;
    
    private volatile ENumber<Long> length;

    private volatile EString lower, trim, upper;

    EStringConst(String constant){
        this.constant = constant;
    }
    
    @Override
    public void accept(Visitor v) {
        v.visit(this);        
    }
    
    @SuppressWarnings("unchecked")
    @Override
    public EString append(Expr<String> s) {
        if (s instanceof Constant){
            return append(((Constant<String>)s).getConstant());
        }else{
            return super.append(s);
        }
    }
    
    @Override
    public EString append(String s) {
        return EStringConst.create(constant + s);
    }
    
    @Override
    public Expr<Character> charAt(int i) {
        return ExprConst.create(constant.charAt(i));
    }

    @Override
    public EString concat(String s) {
        return append(s);
    }
    
    @Override
    public EBoolean eq(String s){
        return EBooleanConst.create(constant.equals(s));
    }
    
    @SuppressWarnings("unchecked")
    @Override
    public boolean equals(Object o) {
        return o instanceof Constant ? ((Constant<?>) o).getConstant().equals(constant) : false;
    }

    @Override
    public EBoolean equalsIgnoreCase(String str) {
        return EBooleanConst.create(constant.equalsIgnoreCase(str));
    }

    @Override
    public String getConstant() {
        return constant;
    }

    @Override
    public int hashCode() {
        return constant.hashCode();
    }
    
    @Override
    public EBoolean isEmpty(){
        return EBooleanConst.create(constant.isEmpty());
    }
    
    @Override
    public EBoolean isNotEmpty(){
        return EBooleanConst.create(!constant.isEmpty());
    }
    
    @Override
    public ENumber<Long> length() {
        if (length == null) {
            length = ENumberConst.create(Long.valueOf(constant.length()));
        }
        return length;
    }
    
    @Override
    public EString lower() {
        if (lower == null) {
            lower = EStringConst.create(constant.toLowerCase());
        }
        return lower;
    }

    @Override
    public EBoolean matches(String pattern){
        return EBooleanConst.create(constant.matches(pattern));
    }
    
    @Override
    public EBoolean ne(String s){
        return EBooleanConst.create(!constant.equals(s));
    }
    
    @SuppressWarnings("unchecked")
    @Override
    public EString prepend(Expr<String> s) {
        if (s instanceof Constant){
            return prepend(((Constant<String>)s).getConstant());
        }else{
            return super.prepend(s);
        }
    }
    
    @Override
    public EString prepend(String s) {
        return EStringConst.create(s + constant);
    }
    
    @Override
    public Expr<String[]> split(String regex) {
        return ExprConst.create(constant.split(regex));
    }
    
    @Override
    public EString substring(int beginIndex) {
        return EStringConst.create(constant.substring(beginIndex));
    }

    @Override
    public EString substring(int beginIndex, int endIndex) {
        return EStringConst.create(constant.substring(beginIndex, endIndex));
    }
    
    @Override
    public EString toLowerCase() {
        return lower();
    }
    
    @Override
    public EString toUpperCase() {
        return upper();
    }
    
    @Override
    public EString trim() {
        if (trim == null) {
            trim = EStringConst.create(constant.trim());
        }
        return trim;
    }
    
    @Override
    public EString upper() {
        if (upper == null){
            upper = EStringConst.create(constant.toUpperCase()); 
        }
        return upper; 
    }
}
