/*
 * Copyright (c) 2009 Mysema Ltd.
 * All rights reserved.
 * 
 */
package com.mysema.query.serialization;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.mysema.commons.lang.Assert;
import com.mysema.query.types.Template;
import com.mysema.query.types.Templates;
import com.mysema.query.types.VisitorBase;
import com.mysema.query.types.custom.Custom;
import com.mysema.query.types.expr.Constant;
import com.mysema.query.types.expr.EArrayConstructor;
import com.mysema.query.types.expr.EConstructor;
import com.mysema.query.types.expr.Expr;
import com.mysema.query.types.operation.Operation;
import com.mysema.query.types.operation.Operator;
import com.mysema.query.types.path.Path;
import com.mysema.query.types.path.PathType;

/**
 * SerializerBase is a stub for Serializer implementations
 * 
 * @author tiwe
 * @version $Id$
 */
public abstract class SerializerBase<SubType extends SerializerBase<SubType>> extends VisitorBase<SubType> {

    private final StringBuilder builder = new StringBuilder();

    protected Map<Object,String> constantToLabel = new HashMap<Object,String>();
    
    protected String constantPrefix = "a";
    
    protected final Templates templates;

    @SuppressWarnings("unchecked")
    private final SubType _this = (SubType) this;

    public SerializerBase(Templates patterns) {
        this.templates = Assert.notNull(patterns,"patterns is null");
    }
    
    public SubType append(String... str) {
        for (String s : str) {
            builder.append(s);
        }
        return _this;
    }

    public Map<Object,String> getConstantToLabel() {
        return constantToLabel;
    }

    public final SubType handle(String sep, List<? extends Expr<?>> expressions) {
        boolean first = true;
        for (Expr<?> expr : expressions) {
            if (!first) {
                append(sep);
            }
            handle(expr);
            first = false;
        }
        return _this;
    }

    public void setConstantPrefix(String prefix){
        this.constantPrefix = prefix;
    }
    
    
    public String toString() {
        return builder.toString();
    }

    @Override
    public void visit(Custom<?> expr) {        
        for (Template.Element element : expr.getTemplate().getElements()){
            if (element.getStaticText() != null){
                append(element.getStaticText());
            }else{
                handle(expr.getArg(element.getIndex()));
            }
        }
    }

    @Override
    public void visit(EArrayConstructor<?> oa) {
        append("new ").append(oa.getElementType().getName()).append("[]{");
        handle(", ", oa.getArgs()).append("}");
    }

    @Override
    public void visit(Constant<?> expr) {        
        if (!constantToLabel.containsKey(expr.getConstant())) {
            String constLabel = constantPrefix + (constantToLabel.size() + 1);
            constantToLabel.put(expr.getConstant(), constLabel);
            append(constLabel);
        } else {
            append(constantToLabel.get(expr.getConstant()));
        }
    }

    @Override
    public void visit(EConstructor<?> expr) {
        append("new ").append(expr.getType().getName()).append("(");
        handle(", ", expr.getArgs()).append(")");
    }

    @Override
    public void visit(Operation<?, ?> expr) {
        visitOperation(expr.getType(), expr.getOperator(), expr.getArgs());
    }

    @Override
    public void visit(Path<?> path) {
        PathType pathType = path.getMetadata().getPathType();
        Template template = templates.getTemplate(pathType);
        List<Expr<?>> args = new ArrayList<Expr<?>>();
        if (path.getMetadata().getParent() != null){
            args.add((Expr<?>) path.getMetadata().getParent());
        }
        args.add(path.getMetadata().getExpression());
        
        for (Template.Element element : template.getElements()){
            if (element.getStaticText() != null){
                append(element.getStaticText()); 
            }else{
                Expr<?> arg = args.get(element.getIndex());
                if (element.isAsString()){
                    append(arg.toString());
                }else if (element.hasConverter()){    
                    handle(element.convert(arg));
                }else{
                    handle(arg);    
                }
            }
        }
    }
    
    @SuppressWarnings("unchecked")
    protected void visitOperation(Class<?> type, Operator<?> operator, List<Expr<?>> args) {
        Template template = templates.getTemplate(operator);
        if (template == null) {
            throw new IllegalArgumentException("Got no pattern for " + operator);
        }
        int precedence = templates.getPrecedence(operator);
        for (Template.Element element : template.getElements()){
            if (element.getStaticText() != null){
                append(element.getStaticText());
            }else if (element.isAsString()){
                append(args.get(element.getIndex()).toString());
            }else{
                int i = element.getIndex();
                boolean wrap = false;
                if (args.get(i) instanceof Operation){
                    wrap = precedence < templates.getPrecedence(((Operation<?, ?>) args.get(i)).getOperator());
                }
                if (wrap){
                    append("(");
                }
                if (element.hasConverter()){
                    handle(element.convert(args.get(i)));
                }else{
                    handle(args.get(i));                    
                }
                if (wrap){
                    append(")");
                }
            }
        }        
    }

}
