/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.tools.util;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;

import org.mule.munit.common.api.util.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <p>
 * This is the class returned by the GetResourceFunction. It has a path that points to the resource wanted, and it has a couple of
 * methods that can convert its content into String, InputStream or byte array
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 3.3.2
 */
public class MunitResource {

  private static Logger logger = LoggerFactory.getLogger(MunitResource.class);
  private String path;
  private ClassLoader classLoader;

  public MunitResource(String path, ClassLoader classLoader) {
    this.path = path;
    this.classLoader = classLoader;
  }

  public InputStream asStream() {
    InputStream streamInput = classLoader.getResourceAsStream(path);
    if (streamInput == null) {
      throw new IllegalArgumentException("The path provided to get the resource does not exist");
    }

    return streamInput;
  }

  public String asString(Charset charset) {
    try {
      return IOUtils.toString(this.asStream(), charset);
    } catch (IOException ioe) {
      logger.error("The file is corrupted");
      return null;
    }
  }

  public byte[] asByteArray() {
    byte[] byteArrayInput = null;
    try {
      byteArrayInput = IOUtils.toByteArray(asStream());
    } catch (IOException ioe) {
      logger.error("The file is corrupted");
    }
    return byteArrayInput;
  }

  public InputStream asReusableStream() {
    InputStream streamInput = this.asStream();
    ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
    try {
      IOUtils.copy(streamInput, outputStream);
    } catch (IOException e) {
      logger.error("The file is corrupted");
    }
    return new ReusableByteArrayInputStream(outputStream.toByteArray());
  }

}
