/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.component.properties;

import static com.google.common.base.Preconditions.checkArgument;
import static java.lang.String.format;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.runtime.api.component.ComponentIdentifier.builder;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.config.api.dsl.model.ConfigurationParameters;
import org.mule.runtime.config.api.dsl.model.ResourceProvider;
import org.mule.runtime.config.api.dsl.model.properties.ConfigurationPropertiesProvider;
import org.mule.runtime.config.api.dsl.model.properties.ConfigurationPropertiesProviderFactory;

/**
 * {@link ConfigurationPropertiesProviderFactory} for a dynamic port property
 * 
 * @since 2.2.0
 * @author Mulesoft Inc.
 */
public class DynamicPortPropertiesProviderFactory implements ConfigurationPropertiesProviderFactory {

  public static final String DYNAMIC_PORT_ELEMENT = "dynamic-port";
  public static final ComponentIdentifier DYNAMIC_PORT =
      builder().namespace("munit").name(DYNAMIC_PORT_ELEMENT).build();
  public static final String PROPERTY_NAME_PARAMETER = "propertyName";
  public static final String MIN_PORT_PARAMETER = "min";
  public static final String MAX_PORT_PARAMETER = "max";

  @Override
  public ComponentIdentifier getSupportedComponentIdentifier() {
    return DYNAMIC_PORT;
  }

  @Override
  public ConfigurationPropertiesProvider createProvider(ConfigurationParameters parameters,
                                                        ResourceProvider externalResourceProvider) {


    String propertyName = parameters.getStringParameter(PROPERTY_NAME_PARAMETER);
    Integer min = getOptionalIntegerParameter(parameters, MIN_PORT_PARAMETER);
    Integer max = getOptionalIntegerParameter(parameters, MAX_PORT_PARAMETER);
    checkArgument(isNotBlank(propertyName),
                  format("Required attribute '%s' of '%s' not found", PROPERTY_NAME_PARAMETER, DYNAMIC_PORT_ELEMENT));

    return new DynamicPortPropertiesProvider(propertyName, min, max);
  }

  private Integer getOptionalIntegerParameter(ConfigurationParameters parameters, String parameterName) {
    String value;
    try {
      value = parameters.getStringParameter(parameterName);
    } catch (Exception e) {
      return null;
    }
    if (value == null) {
      return null;
    }
    try {
      return Integer.valueOf(value);
    } catch (NumberFormatException e) {
      throw new IllegalArgumentException(format("Attribute '%s' must be a valid port number, but was '%s'", parameterName, value),
                                         e);
    }
  }
}
