/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */

package org.mule.munit.remote.runtime.utils;

import java.util.Arrays;
import java.util.stream.Collectors;
import org.mule.munit.common.util.Preconditions;

/**
 * Base class for creating models for Mule artifacts from JSON describer files.
 * <p/>
 * The idea behind this object is to just load some bits of information that later each "loader" will consume to generate things
 * like {@link ClassLoader}, {@link ExtensionModel}, etc.
 *
 * @author Mulesoft Inc.
 * @since 1.0
 */
public abstract class AbstractMuleArtifactModel {

  public static final String NAME = "name";
  public static final String REQUIRED_PRODUCT = "requiredProduct";
  public static final String MIN_MULE_VERSION = "minMuleVersion";
  public static final String ID = "id";
  public static final String CLASS_LOADER_MODEL_LOADER_DESCRIPTOR = "classLoaderModelLoaderDescriptor";
  public static final String BUNDLE_DESCRIPTOR_LOADER = "bundleDescriptorLoader";
  private static final String MANDATORY_FIELD_MISSING_MESSAGE =
      "Invalid artifact descriptor: \"%s\". Mandatory field \"%s\" is missing or has an invalid value. %s";
  private static final String CLASS_LOADER_MODEL_LOADER_DESCRIPTOR_ID = "classLoaderModelLoaderDescriptorid";
  private static final String BUNDLE_DESCRIPTOR_LOADER_ID = "bundleDescriptorLoaderid";
  private String name;
  private String minMuleVersion;
  private Product requiredProduct;
  private MuleArtifactLoaderDescriptor classLoaderModelLoaderDescriptor;
  private MuleArtifactLoaderDescriptor bundleDescriptorLoader;

  protected AbstractMuleArtifactModel(String name, String minMuleVersion, Product requiredProduct,
                                      MuleArtifactLoaderDescriptor classLoaderModelLoaderDescriptor,
                                      MuleArtifactLoaderDescriptor bundleDescriptorLoader) {
    Preconditions.checkArgument(classLoaderModelLoaderDescriptor != null, "classLoaderModelLoaderDescriptor cannot be null");
    Preconditions.checkArgument(bundleDescriptorLoader != null, "bundleDescriptorLoader cannot be null");
    Preconditions.checkArgument(minMuleVersion == null || (new MuleVersion(minMuleVersion)).getRevision() != -1,
                                "descriptor minMuleVersion must have patch version specified");
    this.minMuleVersion = minMuleVersion;
    this.name = name;
    this.requiredProduct = requiredProduct;
    this.classLoaderModelLoaderDescriptor = classLoaderModelLoaderDescriptor;
    this.bundleDescriptorLoader = bundleDescriptorLoader;
  }

  public AbstractMuleArtifactModel() {}

  public String getName() {
    return this.name;
  }

  public String getMinMuleVersion() {
    return this.minMuleVersion;
  }

  public void setMinMuleVersion(String minMuleVersion) {
    this.minMuleVersion = minMuleVersion;
  }

  public Product getRequiredProduct() {
    return this.requiredProduct;
  }

  public MuleArtifactLoaderDescriptor getBundleDescriptorLoader() {
    return this.bundleDescriptorLoader;
  }

  public MuleArtifactLoaderDescriptor getClassLoaderModelLoaderDescriptor() {
    return this.classLoaderModelLoaderDescriptor;
  }

  public void validateModel(String descriptorName) {
    this.validateMandatoryFieldIsSet(descriptorName, this.name, "name");
    this.validateMandatoryFieldIsSet(descriptorName, this.requiredProduct, "requiredProduct", String
        .format("Valid values are %s",
                String.join(", ", (Iterable) Arrays.stream(Product.values()).map(Enum::name).collect(Collectors.toList()))));
    this.validateMandatoryFieldIsSet(descriptorName, this.minMuleVersion, "minMuleVersion");
    this.validateMandatoryFieldIsSet(descriptorName, this.classLoaderModelLoaderDescriptor, "classLoaderModelLoaderDescriptor");
    this.validateMandatoryFieldIsSet(descriptorName, this.classLoaderModelLoaderDescriptor.getId(),
                                     "classLoaderModelLoaderDescriptorid");
    this.validateMandatoryFieldIsSet(descriptorName, this.bundleDescriptorLoader, "bundleDescriptorLoader");
    this.validateMandatoryFieldIsSet(descriptorName, this.bundleDescriptorLoader.getId(), "bundleDescriptorLoaderid");
    this.doValidateCustomFields(descriptorName);
  }

  private void validateMandatoryFieldIsSet(String descriptorName, Object field, String fieldName, String extraErrorMessage) {
    Preconditions.checkState(field != null, String
        .format("Invalid artifact descriptor: \"%s\". Mandatory field \"%s\" is missing or has an invalid value. %s",
                descriptorName, fieldName, extraErrorMessage));
  }

  void validateMandatoryFieldIsSet(String descriptorName, Object field, String fieldName) {
    this.validateMandatoryFieldIsSet(descriptorName, field, fieldName, "");
  }

  void doValidateCustomFields(String descriptorName) {}
}
