/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote.properties.deploy;

import org.mule.munit.remote.api.configuration.RunConfiguration;
import org.mule.munit.remote.runtime.utils.MuleVersion;

import java.util.HashMap;
import java.util.Map;

/**
 * Properties defined for the Mule Container
 ** 
 * @since 2.2.0
 * @author Mulesoft Inc.
 */
public class MuleRuntimeDeploymentProperties implements DeploymentProperties {

  protected static final String PROCESSING_STRATEGY_FACTORY =
      "org.mule.runtime.core.api.processor.strategy.ProcessingStrategyFactory";
  protected static final String PROCESSING_STRATEGY_FACTORY_IMPL =
      "org.mule.runtime.core.internal.processor.strategy.TransactionAwareProactorStreamEmitterProcessingStrategyFactory";
  protected static final String PROCESSING_STRATEGY_MIN_VERSION = "4.2.0";
  protected static final String MULE_DISABLE_ASYNC_RETRY_POLICY_ON_SOURCES = "mule.disableAsyncRetryPolicyOnSources";

  private RunConfiguration runConfiguration;

  public MuleRuntimeDeploymentProperties(RunConfiguration runConfiguration) {
    this.runConfiguration = runConfiguration;
  }

  @Override
  public Map<String, String> get() {
    Map<String, String> properties = new HashMap<>();
    setFactoryProperty(properties);
    setForceSyncReconnection(properties);
    return properties;
  }

  /**
   * Forces sources to retry reconnection synchronously instead of asynchronously which is the default since 4.1.4
   */
  private void setForceSyncReconnection(Map<String, String> properties) {
    if (System.getProperty(MULE_DISABLE_ASYNC_RETRY_POLICY_ON_SOURCES) == null) {
      properties.put(MULE_DISABLE_ASYNC_RETRY_POLICY_ON_SOURCES, Boolean.TRUE.toString());
    }
  }

  /**
   * Sets the default processing strategy property to the implementation used by Mule Flows.
   * 
   * This should be removed with MU-1566
   */
  public void setFactoryProperty(Map<String, String> properties) {
    String runtimeVersion = runConfiguration.getContainerConfiguration().getRuntimeId();
    if (System.getProperty(PROCESSING_STRATEGY_FACTORY) == null) {
      if (isAtLeastMinMuleVersion(runtimeVersion, PROCESSING_STRATEGY_MIN_VERSION)) {
        properties.put(PROCESSING_STRATEGY_FACTORY, PROCESSING_STRATEGY_FACTORY_IMPL);
      }
    }
  }

  public static boolean isAtLeastMinMuleVersion(MuleVersion currentVersion, MuleVersion minMuleVersion) {
    return new MuleVersion(currentVersion.toCompleteNumericVersion()).atLeast(minMuleVersion.toCompleteNumericVersion());
  }

  public static boolean isAtLeastMinMuleVersion(String currentVersion, String minMuleVersion) {
    return isAtLeastMinMuleVersion(new MuleVersion(currentVersion), new MuleVersion(minMuleVersion));
  }


}
