/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote.coverage.server;

import org.mule.munit.common.util.Preconditions;

import static org.mule.munit.remote.coverage.model.CoverageComponentIdentifierReport.CORE_PREFIX;

import java.io.Serializable;

/**
 * Unique identifier for a configuration option. Every configuration option has a namespace and an identifier.
 *
 * The namespace define the extension that defines the component. Even core configuration has a namespace even though there's no
 * namespace used in the configuration files.
 *
 * @author Mulesoft Inc.
 * @since 4.0
 */
// This is public so that DataWeave can get and invoke its methods and not fallback to change the accessibility of its fields
public class DefaultComponentIdentifier implements ComponentIdentifier, Serializable {

  private static final long serialVersionUID = -7904681927277956932L;

  private String namespace;
  private String namespaceLowerCase;
  private String namespaceUri;
  private String name;

  private int hash;

  private DefaultComponentIdentifier() {}

  /**
   * {@inheritDoc}
   */
  @Override
  public String getNamespace() {
    return namespace;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getName() {
    return name;
  }


  /**
   * Unique identifier for a configuration option. Every configuration option has a namespace and an identifier.
   *
   * The namespace define the extension that defines the component. Even core configuration has a namespace even though there's no
   * namespace used in the configuration files.
   *
   * @author Mulesoft Inc.
   * @since 4.0
   */
  public static class Builder implements ComponentIdentifier.Builder {

    private final DefaultComponentIdentifier componentIdentifier = new DefaultComponentIdentifier();

    /**
     * @param namespace namespace identifier of the Mule language extensions module
     * @return the builder
     */
    @Override
    public Builder namespace(String namespace) {
      componentIdentifier.namespace = namespace;
      componentIdentifier.namespaceLowerCase = namespace.toLowerCase();
      return this;
    }

    /**
     * @param identifier identifier unique identifier within the namespace of the language configuration extension
     * @return the builder
     */
    @Override
    public Builder name(String identifier) {
      componentIdentifier.name = identifier;
      return this;
    }

    @Override
    public ComponentIdentifier build() {
      Preconditions.checkState(componentIdentifier.namespace != null && !componentIdentifier.namespace.trim().isEmpty(),
                               "Prefix URI must be not blank");
      Preconditions.checkState(componentIdentifier.name != null && !componentIdentifier.name.trim().isEmpty(),
                               "Name must be not blank");

      componentIdentifier.hash = componentIdentifier.namespaceLowerCase.hashCode();
      componentIdentifier.hash = 31 * componentIdentifier.hash + componentIdentifier.name.hashCode();

      return componentIdentifier;
    }
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    DefaultComponentIdentifier that = (DefaultComponentIdentifier) o;

    if (!namespaceLowerCase.equals(that.namespaceLowerCase)) {
      return false;
    }
    return getName().equals(that.getName());
  }

  @Override
  public int hashCode() {
    return hash;
  }

  /**
   * @return a new {@link Builder}
   */
  public static Builder builder() {
    return new Builder();
  }

  @Override
  public String toString() {
    return getNamespace().equals(CORE_PREFIX) ? getName() : getNamespace() + ":" + getName();
  }


}
