/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.mock.behavior;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.mule.munit.common.behavior.ProcessorCall;
import org.mule.munit.common.exception.MunitError;
import org.mule.munit.common.api.model.EventAttributes;
import org.mule.munit.common.api.model.EventError;
import org.mule.munit.common.api.model.Payload;
import org.mule.munit.common.api.model.Variable;

import org.mule.runtime.api.component.execution.ComponentExecutionException;
import org.mule.runtime.api.event.Event;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.core.api.construct.Flow;


/**
 * The representation of a processor mocked behavior that calls a flow to define which event should be returned.
 *
 * It is used as a replacement of the real processor. We use this in order to know that the processor must return.
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class CallBehaviour extends Behavior {

  private Flow flow;
  private org.mule.munit.common.api.model.Event input;

  /**
   * Create a behaviour for calling a flow when mocking a processor.
   *
   * @param processorCall the processor call to be mocked
   * @param flow the {@link Flow} that will be used
   */
  public CallBehaviour(ProcessorCall processorCall, Flow flow) {
    super(processorCall);
    this.flow = flow;
  }

  /**
   * Invoke the associated flow and store the event, to avoid invoking the flow more than once. Following evaluations will have no
   * effect until {@link #clearEvent()} is invoked.
   *
   * @param input the event that was intercepted during mocking
   *
   * @throws MunitError if {@link Flow#execute(Event)} fails
   */
  public void evaluate(Event input) throws MunitError {
    if (!getEvent().isPresent()) {
      try {
        this.input = buildEvent(input);
        Event output = flow.execute(input).get();
        setEvent(buildEvent(output));

      } catch (ComponentExecutionException cause) {
        setEvent(buildEvent(cause.getEvent()));

      } catch (Throwable cause) {
        if (cause.getCause() instanceof ComponentExecutionException) {
          setEvent(buildEvent(((ComponentExecutionException) cause.getCause()).getEvent()));

        } else {
          clearEvent();

          throw new MunitError(String.format("There was a problem while evaluating '%s'", flow.getName()), cause);
        }
      }
    }
  }

  /**
   * Clear the event set in {@link #evaluate(Event)}. Has no effect if the method was not called, or if the event was already
   * cleared.
   */
  public void clearEvent() {
    setEvent(null);
    this.input = null;
  }

  public Optional<org.mule.munit.common.api.model.Event> getInput() {
    return Optional.ofNullable(input);
  }

  private org.mule.munit.common.api.model.Event buildEvent(Event event) {
    org.mule.munit.common.api.model.Event munitEvent = new org.mule.munit.common.api.model.Event();

    munitEvent.setPayload(buildPayload(event));
    munitEvent.setAttributes(buildAttributes(event));
    munitEvent.setError(buildError(event));
    munitEvent.setVariables(buildVariables(event));

    return munitEvent;
  }

  private Payload buildPayload(Event event) {
    TypedValue<?> corePayload = event.getMessage().getPayload();
    MediaType coreMediaType = corePayload.getDataType().getMediaType();

    Payload munitPayload = new Payload();
    munitPayload.setValue(corePayload.getValue());
    munitPayload.setMediaType(coreMediaType.withoutParameters().toString());
    coreMediaType.getCharset().ifPresent(charset -> munitPayload.setEncoding(charset.toString()));

    return munitPayload;
  }

  private EventAttributes buildAttributes(Event event) {
    TypedValue<?> coreAttributes = event.getMessage().getAttributes();

    if (coreAttributes != null) {
      EventAttributes munitAttributes = new EventAttributes();
      MediaType coreMediaType = coreAttributes.getDataType().getMediaType();

      munitAttributes.setValue(coreAttributes.getValue());
      munitAttributes.setMediaType(coreMediaType.withoutParameters().toString());
      coreMediaType.getCharset().ifPresent(charset -> munitAttributes.setEncoding(charset.toString()));

      return munitAttributes;
    }

    return null;
  }

  private EventError buildError(Event event) {
    if (event.getError().isPresent()) {
      EventError munitError = new EventError();
      munitError.setCause(event.getError().get().getCause());
      munitError.setTypeId(event.getError().get().getErrorType().getIdentifier());

      return munitError;
    }

    return null;
  }

  private List<Variable> buildVariables(Event event) {
    Map<String, TypedValue<?>> coreEventVariables = event.getVariables();
    List<Variable> munitVariables = new ArrayList<>();
    for (String key : coreEventVariables.keySet()) {
      Variable munitVariable = new Variable();
      munitVariable.setKey(key);
      munitVariable.setValue(coreEventVariables.get(key).getValue());
      coreEventVariables.get(key).getDataType().getMediaType().getCharset()
          .ifPresent(charset -> munitVariable.setEncoding(charset.toString()));
      MediaType mimeType = coreEventVariables.get(key).getDataType().getMediaType();

      if (!mimeType.equals(MediaType.ANY)) {
        munitVariable.setMediaType(mimeType.withoutParameters().toString());
      }

      munitVariables.add(munitVariable);
    }

    return munitVariables;
  }

  public CallBehaviour copyWithoutState() {
    return new CallBehaviour(getProcessorCall(), getFlow());
  }

  public Flow getFlow() {
    return flow;
  }
}
