/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.mock.interception;

import org.mule.munit.common.api.event.DataTypeBuilder;
import org.mule.munit.common.api.event.MessageBuilderDecorator;
import org.mule.munit.common.api.model.Event;
import org.mule.munit.common.api.model.EventAttributes;
import org.mule.munit.common.api.model.NullObject;
import org.mule.munit.common.api.model.Payload;
import org.mule.runtime.api.interception.InterceptionEvent;
import org.mule.runtime.api.message.Message;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.DataTypeParamsBuilder;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;

import java.nio.charset.Charset;

import org.apache.commons.lang3.StringUtils;

/**
 * The objective of this class is to mutate the received InterceptionEvent.
 * <p>
 * The nature of it is such that it can be mutated and there is no need to generate a new one.
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class InterceptingEventBuilder {

  public InterceptionEvent build(InterceptionEvent originalEvent, Event mockedEvent) {
    setVariables(originalEvent, mockedEvent);

    originalEvent.message(buildMessage(originalEvent.getMessage(), mockedEvent));

    return originalEvent;
  }

  private void setVariables(InterceptionEvent originalEvent, Event mockedEvent) {
    if (null != mockedEvent.getVariables()) {
      mockedEvent.getVariables()
          .forEach(v -> originalEvent.addVariable(v.getKey(), v.getValue(), DataTypeBuilder.fromTypedKeyValue(v).build()));
    }
  }

  private TypedValue getAttributes(EventAttributes mockedAttributes) {
    DataTypeParamsBuilder dataTypeParamsBuilder = DataType.builder().fromObject(mockedAttributes.getValue());
    if (StringUtils.isNotBlank(mockedAttributes.getMediaType())) {
      dataTypeParamsBuilder.mediaType(mockedAttributes.getMediaType());
    }
    dataTypeParamsBuilder.charset(mockedAttributes.getEncoding());
    return new TypedValue<>(mockedAttributes.getValue(), dataTypeParamsBuilder.build());
  }

  private Message buildMessage(Message originalMessage, Event mockedEvent) {
    MessageBuilderDecorator builder = new MessageBuilderDecorator(originalMessage);
    Payload mockedEventPayload = mockedEvent.getPayload();
    EventAttributes mockedAttributes = mockedEvent.getAttributes();
    boolean mockedEventPayloadExists = !(mockedEventPayload == null || mockedEventPayload.getValue() instanceof NullObject);
    boolean mockedAttributesExists = mockedAttributes != null && !(mockedAttributes.getValue() instanceof NullObject);

    if (mockedEventPayloadExists) {
      builder.withPayload(mockedEventPayload.getValue());
      if (!mockedAttributesExists) {
        builder.withAttributes(TypedValue.of(null));
      }

      if (mockedEventPayload.getMediaType() != null) {
        MediaType mediaType = MediaType.parse(mockedEventPayload.getMediaType());
        if (mockedEventPayload.getEncoding() != null) {
          mediaType = mediaType.withCharset(Charset.forName(mockedEventPayload.getEncoding()));
        }
        builder.withMediaType(mediaType);
      }
    }

    if (mockedAttributesExists) {
      TypedValue typedValueAttributes = getAttributes(mockedAttributes);
      builder.withAttributes(typedValueAttributes);
      if (!mockedEventPayloadExists) {
        builder.withPayload(null);
      }
    }

    return builder.build();
  }

}
