/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugin.maven.util;

import static java.lang.String.format;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.joining;

import org.mule.munit.plugin.maven.locators.RemoteRepositoriesLocator;
import org.mule.runtime.api.deployment.meta.Product;

import com.mulesoft.anypoint.discovery.api.RuntimeVersionProvider;
import com.mulesoft.anypoint.discovery.api.exception.NoLatestVersionFoundException;
import com.mulesoft.anypoint.discovery.api.exception.VersionDiscoveryException;
import com.mulesoft.anypoint.discovery.core.aether.AetherRuntimeVersionProvider;
import com.mulesoft.anypoint.discovery.core.version.MuleArtifactVersionFactory;
import com.mulesoft.anypoint.discovery.core.version.MuleArtifactVersionValidator;

import java.util.List;
import java.util.Optional;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.logging.Log;
import org.eclipse.aether.RepositorySystem;
import org.eclipse.aether.RepositorySystemSession;
import org.eclipse.aether.artifact.DefaultArtifact;
import org.eclipse.aether.repository.RemoteRepository;
import org.eclipse.aether.resolution.VersionRangeRequest;
import org.eclipse.aether.resolution.VersionRangeResolutionException;
import org.eclipse.aether.resolution.VersionRangeResult;
import org.eclipse.aether.version.Version;

/**
 * <p>
 * Creates {@link RuntimeVersionProvider} based on the Mule runtime product
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class RuntimeVersionProviderFactory {

  private final Log log;
  private final RepositorySystem repositorySystem;
  private final RepositorySystemSession repositorySystemSession;
  private final RemoteRepositoriesLocator remoteRepositoriesLocator;

  public RuntimeVersionProviderFactory(RepositorySystem repositorySystem, RepositorySystemSession repositorySystemSession,
                                       RemoteRepositoriesLocator remoteRepositoriesLocator, Log log) {
    this.repositorySystem = repositorySystem;
    this.repositorySystemSession = repositorySystemSession;
    this.remoteRepositoriesLocator = remoteRepositoriesLocator;
    this.log = log;
  }

  public Optional<RuntimeVersionProvider> create(Product product) throws MojoExecutionException {
    List<RemoteRepository> remoteRepositories = remoteRepositoriesLocator.locate();
    try {
      validateRepositories(remoteRepositories);
      return of(new AetherRuntimeVersionProvider(repositorySystem, repositorySystemSession, remoteRepositories,
                                                 getProduct(product), new MuleArtifactVersionFactory(),
                                                 new MuleArtifactVersionValidator()));
    } catch (VersionDiscoveryException e) {
      throw new MojoExecutionException("An error occurred while discovering versions for product " + product
          + " in the following repositories " + remoteRepositories);
    } catch (NoLatestVersionFoundException e) {
      log.warn("Unable to find Mule Runtime versions for product " + product);
      return Optional.empty();
    }
  }

  public List<Version> muleFrameworkVersions() throws MojoExecutionException {
    return muleFrameworkVersionsRange().getVersions();
  }

  public Optional<Version> muleFrameworkLatestVersion() throws MojoExecutionException {
    return ofNullable(muleFrameworkVersionsRange().getHighestVersion());
  }

  private VersionRangeResult muleFrameworkVersionsRange() throws MojoExecutionException {
    List<RemoteRepository> remoteRepositories = remoteRepositoriesLocator.locate();
    VersionRangeRequest rangeRequest =
        new VersionRangeRequest(new DefaultArtifact("org.mule.fwk:mule-framework-bom:(0,]"), remoteRepositories,
                                null);

    VersionRangeResult versionRange;
    try {
      versionRange = repositorySystem.resolveVersionRange(repositorySystemSession, rangeRequest);
    } catch (VersionRangeResolutionException e) {
      throw new MojoExecutionException("An error occurred while discovering versions for MuleFramework in the following repositories "
          + remoteRepositories);
    }
    return versionRange;
  }

  private com.mulesoft.anypoint.discovery.core.aether.Product getProduct(Product product) {
    switch (product) {
      case MULE_EE:
        return com.mulesoft.anypoint.discovery.core.aether.Product.EE_BOM;
      case MULE:
        return com.mulesoft.anypoint.discovery.core.aether.Product.CE_BOM;
      default:
        throw new IllegalArgumentException("Unsupported product to discover runtimes: " + product);
    }
  }

  private void validateRepositories(List<RemoteRepository> remoteRepositories) throws MojoExecutionException {
    if (remoteRepositories.isEmpty()) {
      throw new MojoExecutionException(format("%nNo repositories found to discover runtimes.%nFound:%n  %s%nbut needed one of:%n  %s%n",
                                              remoteRepositoriesLocator.locateAll().stream().map(RemoteRepository::getUrl)
                                                  .collect(joining(format(",%n  "))),
                                              String.join(format(",%n  "),
                                                          remoteRepositoriesLocator.getWhitelistedRepositories())));
    }
  }
}
