/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugin.maven.util;

import static org.mule.munit.remote.FolderNames.MUNIT;
import static org.mule.munit.remote.FolderNames.TEST_MULE;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;

import static org.mule.munit.common.util.Preconditions.checkNotNull;
import static org.apache.commons.lang3.StringUtils.EMPTY;

import org.mule.maven.client.internal.DefaultSettingsSupplierFactory;
import org.mule.maven.client.internal.MavenEnvironmentVariables;
import org.mule.munit.plugin.maven.locators.Log4J2ConfigurationLocator;
import org.mule.munit.plugin.maven.locators.TestSuiteFilesLocator;
import org.mule.munit.plugin.maven.runner.structure.WorkingDirectoryGenerator;
import org.mule.munit.plugin.maven.runtime.TargetProduct;
import org.mule.munit.remote.MinMuleVersionConfig;
import org.mule.munit.remote.api.configuration.NotifierConfiguration.NotifierConfigurationBuilder;
import org.mule.munit.remote.notifiers.SocketNotifier;
import org.mule.munit.remote.api.configuration.ContainerConfiguration;
import org.mule.munit.remote.api.configuration.MavenConfiguration;
import org.mule.munit.remote.api.configuration.NotifierConfiguration;
import org.mule.munit.remote.api.configuration.NotifierParameter;
import org.mule.munit.remote.api.configuration.Repository;
import org.mule.munit.remote.api.configuration.RunConfiguration;
import org.mule.munit.remote.notifiers.ConsoleNotifier;

import java.io.File;
import java.net.URI;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.apache.maven.execution.MavenSession;
import org.apache.maven.model.Profile;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.MavenProject;

/**
 * Creates {@link RunConfiguration} based on basic MUnit parameters
 *
 * @author Mulesoft Inc.
 * @since 2.2.0
 */
public abstract class BaseRunConfigurationFactory implements RunConfigurationFactory {

  public static final String RUN_TOKEN_CONSOLE_PARAMETER = "runToken";
  public static final String PORT_PARAMETER = "port";

  private final Log log;
  private final String munitTags;
  private final String projectName;
  private final Boolean skipAfterFailure;
  private final TestSuiteFileFilter testSuiteFileFilter;
  private final WorkingDirectoryGenerator workingDirectoryGenerator;
  private final TargetProduct targetProduct;
  private final Boolean clearParameters;

  private final File munitSrcFolder;
  private final URI defaultMunitSrcFolder;
  protected String runToken;

  private final MavenProject project;
  private final MavenSession session;
  private final List<String> mmvs;
  private final MinMuleVersionConfig minMuleVersionConfig;
  private final String runtimeLocalDistribution;

  public BaseRunConfigurationFactory(Log log, String projectName, String munitTest, String munitTags, Boolean skipAfterFailure,
                                     TargetProduct targetProduct, WorkingDirectoryGenerator workingDirectoryGenerator,
                                     File munitSrcFolder, MavenProject project, MavenSession session, Boolean clearParameters,
                                     List<String> mmvs, MinMuleVersionConfig minMuleVersionConfig,
                                     String runtimeLocalDistribution) {


    checkNotNull(log, "The log must not be null");
    checkNotNull(skipAfterFailure, "The skipAfterFailure must not be null");
    checkNotNull(targetProduct, "The muleApplicationModelLoader must not be null nor empty");
    checkNotNull(workingDirectoryGenerator, "The WorkingDirectoryGenerator must not be null");
    checkNotNull(munitSrcFolder, "The munitSrcFolder must not be null");
    checkNotNull(project, "The project must not be null");
    checkNotNull(session, "The session must not be null");

    this.log = log;
    this.projectName = projectName;
    this.munitTags = munitTags;
    this.skipAfterFailure = skipAfterFailure;
    this.targetProduct = targetProduct;
    this.testSuiteFileFilter = new TestSuiteFileFilter(log, munitTest);
    this.clearParameters = clearParameters;
    this.munitSrcFolder = munitSrcFolder;
    this.defaultMunitSrcFolder = Paths.get(project.getBuild().getDirectory(), TEST_MULE.value(), MUNIT.value()).toUri();
    this.runToken = UUID.randomUUID().toString();
    this.workingDirectoryGenerator = workingDirectoryGenerator;
    this.mmvs = mmvs;
    this.project = project;
    this.session = session;
    this.minMuleVersionConfig = minMuleVersionConfig;
    this.runtimeLocalDistribution = runtimeLocalDistribution;
  }

  @Override
  public RunConfiguration create() throws MojoExecutionException {
    return getRunConfigurationBuilder().build();
  }

  protected MavenConfiguration.MavenConfigurationBuilder getMavenConfigurationBuilder() {
    DefaultSettingsSupplierFactory defaultSettingsSupplierFactory =
        new DefaultSettingsSupplierFactory(new MavenEnvironmentVariables());

    MavenConfiguration.MavenConfigurationBuilder mavenConfigurationBuilder = new MavenConfiguration.MavenConfigurationBuilder();
    mavenConfigurationBuilder.withMavenRepositoryDirectoryPath(session.getLocalRepository().getBasedir())
        .withSettingsXmlFilePath(session.getRequest().getUserSettingsFile().getAbsolutePath())
        .withGlobalSettingsXmlFilePath(session.getRequest().getGlobalSettingsFile().getAbsolutePath())
        .withForcePolicyUpdate(true)
        .withActiveProfiles(project.getActiveProfiles().stream().map(Profile::getId).collect(toList()))
        .withOfflineMode(session.isOffline())
        .withIgnoreArtifactDescriptorRepositories(false)
        .withRemoteRepositories(getRemoteRepositories());
    defaultSettingsSupplierFactory.environmentSettingsSecuritySupplier()
        .ifPresent(file -> mavenConfigurationBuilder.withSecuritySettingsFilePath(file.getPath()));
    return mavenConfigurationBuilder;
  }

  protected RunConfiguration.RunConfigurationBuilder getRunConfigurationBuilder() throws MojoExecutionException {
    List<RunConfiguration.Test> testNamesWithSuite = testSuiteFileFilter.getTestNameRegEx();
    return new RunConfiguration.RunConfigurationBuilder().withRunToken(runToken)
        .withProjectName(projectName)
        .withTags(munitTags == null ? Collections.emptySet() : new HashSet<>(Arrays.asList(munitTags.split(","))))
        .withSkipAfterFailure(skipAfterFailure)
        .withClearParameters(clearParameters)
        .withTestNames(testNamesWithSuite.stream().map(RunConfiguration.Test::getName).collect(toSet()))
        .withTestNamesWithSuite(testNamesWithSuite)
        .withSuitePaths(locateMunitTestSuitesToRun())
        .withAllSuitePaths(locateAllMunitTestSuites())
        .withNotifierConfigurations(getNotifierConfigurations())
        .withRunToken(runToken)
        .withMmvs(mmvs)
        .withMinMuleVersionConfig(minMuleVersionConfig)
        .withContainerConfiguration(getContainerConfigurationBuilder().build())
        .withDomainLocation(workingDirectoryGenerator.generateDomainStructure()
            .map(domainPath -> domainPath.toFile().getAbsolutePath())
            .orElse(StringUtils.EMPTY));
  }

  protected List<NotifierConfiguration> getNotifierConfigurations() {
    List<NotifierParameter> consoleParameters = new ArrayList<>();
    consoleParameters.add(new NotifierParameter(RUN_TOKEN_CONSOLE_PARAMETER, String.class.getCanonicalName(), runToken));
    NotifierConfiguration consoleNotifierConfiguration = new NotifierConfiguration.NotifierConfigurationBuilder()
        .withClazz(ConsoleNotifier.class.getCanonicalName())
        .withParameters(consoleParameters)
        .build();

    List<NotifierConfiguration> notifierConfigurations = new ArrayList<>();
    notifierConfigurations.add(consoleNotifierConfiguration);
    if (getSocketPort() != null) {
      NotifierConfigurationBuilder notifierConfigurationBuilder = new NotifierConfigurationBuilder();
      notifierConfigurationBuilder.withClazz(SocketNotifier.class.getCanonicalName());
      NotifierParameter portNotifierParameter =
          new NotifierParameter(PORT_PARAMETER, String.class.getCanonicalName(), getSocketPort());
      NotifierParameter tokenNotifierParameter =
          new NotifierParameter(RUN_TOKEN_CONSOLE_PARAMETER, String.class.getCanonicalName(), runToken);
      notifierConfigurationBuilder.withParameters(Arrays.asList(portNotifierParameter, tokenNotifierParameter));
      notifierConfigurations.add(notifierConfigurationBuilder.build());
    }
    return notifierConfigurations;
  }

  protected abstract String getSocketPort();

  protected abstract ContainerConfiguration.ContainerConfigurationBuilder getContainerConfigurationBuilder();

  protected Set<String> locateAllMunitTestSuites() {
    return new TestSuiteFilesLocator().locateFiles(munitSrcFolder).stream()
        .map(suiteFile -> defaultMunitSrcFolder.relativize(suiteFile.toURI()).getPath())
        .collect(toSet());
  }

  protected Set<String> locateMunitTestSuitesToRun() {
    return locateAllMunitTestSuites().stream().filter(testSuiteFileFilter::shouldFilter).collect(toSet());
  }

  protected String getLog4JConfigurationFilePath() {
    return getLog4JConfigurationFile().isPresent() ? getLog4JConfigurationFile().get().getAbsolutePath() : EMPTY;
  }

  private Optional<File> getLog4JConfigurationFile() {
    List<File> files = new Log4J2ConfigurationLocator(log).locateFiles(new File(project.getBuild().getDirectory()));
    return files.isEmpty() ? Optional.empty() : Optional.of(files.get(0));
  }

  private List<Repository> getRemoteRepositories() {
    return project.getRemotePluginRepositories().stream().map(r -> new Repository(r.getId(), r.getUrl()))
        .collect(Collectors.toList());
  }

  public WorkingDirectoryGenerator getWorkingDirectoryGenerator() {
    return workingDirectoryGenerator;
  }

  public TargetProduct getTargetProduct() {
    return targetProduct;
  }

  public String getRuntimeLocalDistribution() {
    return runtimeLocalDistribution;
  }
}
