/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugin.maven.runtime;

import static org.mule.munit.common.util.Preconditions.checkArgument;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import org.mule.runtime.api.meta.MuleVersion;

import java.util.Objects;

import org.apache.commons.lang3.StringUtils;

/**
 * Version and product
 *
 * @since 2.4.0
 * @author Mulesoft Inc.
 */
public class TargetProduct implements Comparable<TargetProduct> {

  private static final String TARGET_PRODUCT_SEPARATOR = ":";
  private final MuleVersion version;
  private final Product product;

  public TargetProduct(String version, Product product) {
    checkArgument(isNotBlank(version), "Version cannot be blank");
    this.version = new MuleVersion(version);

    Objects.requireNonNull(product, "Product cannot be blank");

    this.product = product;
  }

  public static TargetProduct parse(String targetProduct) {
    String[] versionAndProduct = StringUtils.split(targetProduct, TARGET_PRODUCT_SEPARATOR);
    if (versionAndProduct.length != 2) {
      throw new IllegalArgumentException("Target product should be in the form of <product>:<version>. For example MULE_EE:4.1.1");
    }
    String product = versionAndProduct[0];
    String version = versionAndProduct[1];
    return new TargetProduct(version, Product.valueOf(product));
  }

  public Product getProduct() {
    return product;
  }

  public String getVersion() {
    return version.toString();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }

    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    TargetProduct that = (TargetProduct) o;
    return this.version.equals(that.version) && this.product.equals(that.product);
  }

  @Override
  public int hashCode() {
    int result = version != null ? version.hashCode() : 0;
    result = 31 * result + (product != null ? product.hashCode() : 0);
    return result;
  }

  @Override
  public String toString() {
    return product + TARGET_PRODUCT_SEPARATOR + version.toString();
  }

  @Override
  public int compareTo(TargetProduct that) {
    if (that == null) {
      throw new NullPointerException("Can't compare against null");
    }

    if (this.equals(that)) {
      return 0;
    }

    if (this.product.ordinal() == that.product.ordinal()) {
      return this.version.newerThan(that.version) ? +1 : -1;
    } else {
      return this.product.ordinal() - that.product.ordinal();
    }
  }
}
