/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugin.maven.runner.printer;

import static java.lang.String.format;

import static org.apache.commons.lang3.StringUtils.EMPTY;

import org.mule.munit.plugin.maven.report.xml.sonar.SonarFile;
import org.mule.munit.plugin.maven.report.xml.sonar.SonarTestCase;
import org.mule.munit.plugin.maven.report.xml.sonar.SonarTestExecutions;
import org.mule.munit.plugin.maven.runner.model.RunResult;
import org.mule.munit.plugin.maven.runner.model.SuiteResult;
import org.mule.munit.plugin.maven.runner.model.TestResult;
import org.mule.munit.plugin.maven.runtime.TargetProduct;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.PrintStream;
import java.io.UnsupportedEncodingException;

import com.thoughtworks.xstream.XStream;

import org.apache.maven.plugin.logging.Log;

/**
 * <p>
 * SonarResultFilePrinter
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.3.0
 */
public class SonarResultFilePrinter extends FileResultPrinter {

  private static final String MUNIT_PATH = "src/test/munit";

  public SonarResultFilePrinter(File reportBasePath, Log log) {
    super(reportBasePath, log);
  }

  @Override
  public void print(TargetProduct targetProduct, RunResult runResult) {
    try {
      for (SuiteResult suiteResult : runResult.getSuites()) {
        printSuiteResult(targetProduct, suiteResult);
      }
    } catch (FileNotFoundException | UnsupportedEncodingException e) {
      e.printStackTrace();
    }
  }

  public void printSuiteResult(TargetProduct targetProduct, SuiteResult result)
      throws FileNotFoundException, UnsupportedEncodingException {

    String parameterization = result.getParameterization().map(p -> ".[" + p + "]").orElse(EMPTY);
    String name =
        String.format("%s.%s%s.xml", targetProduct.getProduct().name(), targetProduct.getVersion(), parameterization);


    SonarTestExecutions sonarTestExecutions = readExistReport(name);

    String suitePath = new File(MUNIT_PATH, result.getSuitePath()).getPath();
    SonarFile suite = new SonarFile(suitePath);
    sonarTestExecutions.add(suite);

    for (TestResult testResult : result.getTests()) {
      printTestResult(testResult, suite);
    }

    XStream xStream = createSecureXStream();
    PrintStream out = getResultPrintStream(format("%s", name));
    out.print(xStream.toXML(sonarTestExecutions));
  }

  public void printTestResult(TestResult testResult, SonarFile suite) {
    SonarTestCase testCase = new SonarTestCase(testResult.getTime(), testResult.getTestName());

    if (testResult.isIgnored()) {
      testCase.setSkipped(testResult.getCause());
    } else if (testResult.hasFailed()) {
      testCase.setFailure(testResult.getCause());
    } else if (testResult.hasError()) {
      testCase.setError(testResult.getCause());
    }

    suite.add(testCase);
  }

  private SonarTestExecutions readExistReport(String fileName) {
    File report = new File(this.reportBasePath, fileName);

    if (!report.exists()) {
      return new SonarTestExecutions();
    }

    try {
      XStream xStream = createSecureXStream();
      SonarTestExecutions sonarReport = (SonarTestExecutions) xStream.fromXML(report);
      return sonarReport;

    } catch (Exception e) {
      e.printStackTrace();
    }

    return new SonarTestExecutions();
  }

  private XStream createSecureXStream() {
    XStream xStream = new XStream();
    XStream.setupDefaultSecurity(xStream); // to be removed after 1.5
    xStream.allowTypesByWildcard(new String[] {
        "org.mule.**"
    });
    xStream.alias("testExecutions", SonarTestExecutions.class);
    xStream.autodetectAnnotations(true);

    return xStream;
  }

}
