/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugin.maven.locators;

import static org.mule.munit.plugin.maven.runtime.Product.MULE_FWK;
import static org.mule.runtime.api.deployment.meta.Product.MULE;
import static org.mule.runtime.api.deployment.meta.Product.MULE_EE;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toCollection;

import org.mule.munit.plugin.maven.runtime.Product;
import org.mule.munit.plugin.maven.runtime.TargetProduct;
import org.mule.munit.plugin.maven.util.RuntimeVersionProviderFactory;
import org.mule.runtime.api.meta.MuleVersion;

import com.mulesoft.anypoint.discovery.api.RuntimeVersionProvider;
import com.mulesoft.anypoint.discovery.api.version.ArtifactVersion;

import java.util.HashMap;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import java.util.stream.Stream;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.logging.Log;
import org.eclipse.aether.version.Version;


/**
 * <p>
 * Locates all product versions and obtains resulting {@link TargetProduct} list
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.4.0
 */
public class ProductVersionsLocator {

  private final RuntimeVersionProviderFactory runtimeVersionProviderFactory;
  private Product product;
  private String minVersion;
  private boolean includeSnapshots;
  private final Log log;
  private boolean useLatestPatches;

  public ProductVersionsLocator(RuntimeVersionProviderFactory runtimeVersionProviderFactory, Log log) {
    this.runtimeVersionProviderFactory = runtimeVersionProviderFactory;
    this.log = log;
  }

  public ProductVersionsLocator withMinVersion(String minVersion) {
    this.minVersion = minVersion;
    return this;
  }

  public ProductVersionsLocator withProduct(Product product) {
    this.product = product;
    return this;
  }

  public ProductVersionsLocator includingSnapshots(boolean includeSnapshots) {
    this.includeSnapshots = includeSnapshots;
    return this;
  }

  public ProductVersionsLocator usingLatestPatches(boolean useLatestPatches) {
    this.useLatestPatches = useLatestPatches;
    return this;
  }

  public Set<TargetProduct> locate() throws MojoExecutionException {
    Stream<TargetProduct> targetRuntimes = Stream.empty();

    if (product.equals(Product.MULE_CE) || product.equals(Product.MULE_EE)) {
      List<ArtifactVersion> eeRuntimeVersions =
          product.supportsEe() ? locateRuntimes(runtimeVersionProviderFactory, MULE_EE) : emptyList();
      List<ArtifactVersion> ceRuntimeVersions =
          product.supportsCe() ? locateRuntimes(runtimeVersionProviderFactory, MULE) : emptyList();

      targetRuntimes = createTargetProducts(eeRuntimeVersions, ceRuntimeVersions, emptyList());
    } else if (product.equals(Product.MULE_FWK)) {
      List<Version> muleFwkVersions = runtimeVersionProviderFactory.muleFrameworkVersions();

      targetRuntimes = createTargetProducts(emptyList(), emptyList(), muleFwkVersions);
    }

    targetRuntimes = targetRuntimes
        .filter(this::isNotCePatchVersion)
        .filter(this::includeSnapshots)
        .filter(this::greaterThanMinVersion);

    if (this.useLatestPatches) {
      targetRuntimes = filterLatestPatches(targetRuntimes);
    }

    log.debug("Discovered Products: " + targetRuntimes);
    return targetRuntimes.collect(toCollection(TreeSet::new));
  }

  private boolean isNotCePatchVersion(TargetProduct targetProduct) {
    if (targetProduct.getProduct() == Product.MULE_CE) {
      MuleVersion muleVersion = new MuleVersion(targetProduct.getVersion());
      return muleVersion.getRevision() == 0 || muleVersion.equals(product.getMinVersion());
    }
    return true;
  }

  private Stream<TargetProduct> createTargetProducts(List<ArtifactVersion> eeVersions,
                                                     List<ArtifactVersion> ceVersions,
                                                     List<Version> muleFwkVersions) {
    Stream<TargetProduct> ceRuntimes = ceVersions.stream().map(version -> toTargetProduct(version, MULE));
    Stream<TargetProduct> eeRuntimes = eeVersions.stream().map(version -> toTargetProduct(version, MULE_EE));
    Stream<TargetProduct> muleFrameworks =
        muleFwkVersions.stream().map(version -> new TargetProduct(version.toString(), MULE_FWK));
    return Stream.concat(Stream.concat(ceRuntimes, eeRuntimes), muleFrameworks);
  }

  private List<ArtifactVersion> locateRuntimes(RuntimeVersionProviderFactory versionProviderFactory,
                                               org.mule.runtime.api.deployment.meta.Product product)
      throws MojoExecutionException {
    return versionProviderFactory.create(product).map(RuntimeVersionProvider::all).orElse(emptyList());
  }

  private boolean greaterThanMinVersion(TargetProduct targetRuntime) {
    return new MuleVersion(targetRuntime.getVersion()).atLeast(minVersion);
  }

  private boolean includeSnapshots(TargetProduct targetRuntime) {
    MuleVersion muleVersion = new MuleVersion(targetRuntime.getVersion());
    if (muleVersion.hasSuffix()) {
      return includeSnapshots && "SNAPSHOT".equals(muleVersion.getSuffix());
    }
    return true;
  }

  private TargetProduct toTargetProduct(ArtifactVersion artifactVersion, org.mule.runtime.api.deployment.meta.Product product) {
    return new TargetProduct(artifactVersion.value(),
                             product == org.mule.runtime.api.deployment.meta.Product.MULE
                                 ? Product.MULE_CE
                                 : Product.valueOf(product.name()));
  }

  private Stream<TargetProduct> filterLatestPatches(Stream<TargetProduct> targetRuntimes) {

    HashMap<String, TargetProduct> latestPatches = new HashMap<>();

    targetRuntimes.forEach(targetRuntime -> {
      MuleVersion version = new MuleVersion(targetRuntime.getVersion());

      String versionKey = String.format("%d.%d%s", version.getMajor(), version.getMinor(), version.getSuffix());

      if (latestPatches.containsKey(versionKey)) {
        TargetProduct currentLatest = latestPatches.get(versionKey);

        if (version.newerThan(currentLatest.getVersion())) {
          latestPatches.put(versionKey, targetRuntime);
        }
      } else {
        latestPatches.put(versionKey, targetRuntime);
      }

    });

    return latestPatches.values().stream();
  }
}
