/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.extension.maven.internal.generator;

import static org.mule.munit.remote.FolderNames.MAVEN;
import static org.mule.munit.remote.FolderNames.MULE_ARTIFACT;
import static org.mule.munit.remote.FolderNames.MUNIT;
import static org.mule.munit.remote.FolderNames.TEST_CLASSES;
import static org.mule.munit.remote.FolderNames.TEST_MULE;
import static org.mule.munit.plugin.maven.project.MuleApplicationStructureGenerator.RUN_CONFIGURATION_JSON;
import static org.mule.tools.api.packager.structure.FolderNames.MAIN;
import static org.mule.tools.api.packager.structure.FolderNames.META_INF;
import static org.mule.tools.api.packager.structure.FolderNames.MULE_SRC;
import static org.mule.tools.api.packager.structure.FolderNames.REPOSITORY;
import static org.mule.tools.api.packager.structure.FolderNames.SRC;
import static org.mule.tools.api.packager.structure.FolderNames.TARGET;
import static org.mule.tools.api.packager.structure.PackagerFiles.MULE_ARTIFACT_JSON;
import static org.mule.tools.api.packager.structure.PackagerFiles.POM_PROPERTIES;
import static org.mule.tools.api.packager.structure.PackagerFiles.POM_XML;

import static java.nio.charset.Charset.defaultCharset;
import static java.nio.file.Files.createTempDirectory;
import static java.util.Objects.nonNull;

import static com.google.common.base.Preconditions.checkArgument;
import static org.apache.commons.io.FileUtils.writeStringToFile;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import org.mule.munit.extension.maven.internal.generator.maven.ApplicationPomGenerator;
import org.mule.munit.remote.api.configuration.RunConfiguration;
import org.mule.munit.plugin.maven.project.ApplicationStructureGenerator;
import org.mule.tools.api.packager.archiver.MuleArchiver;
import org.mule.tools.api.packager.archiver.MuleExplodedArchiver;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;

import org.apache.commons.io.FileUtils;
import org.apache.maven.model.Model;
import org.apache.maven.model.io.xpp3.MavenXpp3Writer;
import org.apache.maven.project.MavenProject;

/**
 * Creates an application structure based on an extensions build directory
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class ExtensionApplicationStructureGenerator implements ApplicationStructureGenerator {

  private final Path originFolder;
  private final ApplicationPomGenerator applicationPomGenerator;
  private final byte[] pomPropertiesContent;
  private final byte[] muleArtifactJson;
  private final String groupId;
  private final String artifactId;
  private final File tempDir;
  private File sourcesFolder;
  private MuleArchiver archiver;
  private ApplicationRepositoryGenerator repositoryGenerator;

  public ExtensionApplicationStructureGenerator(Path originFolder, String groupId, String artifactId,
                                                ApplicationPomGenerator applicationPomGenerator,
                                                byte[] pomPropertiesContent, byte[] muleArtifactJsonContent)
      throws IOException {
    checkArgument(originFolder != null, "Origin folder cannot be null");
    checkArgument(originFolder.toFile().exists(), "Origin folder should exist");
    checkArgument(isNotBlank(groupId), "Group Id cannot be blank");
    checkArgument(isNotBlank(artifactId), "Artifact Id cannot be blank");
    checkArgument(applicationPomGenerator != null, "Application pom generator cannot be null");
    checkArgument(pomPropertiesContent != null, "Pom properties content cannot be null");
    checkArgument(muleArtifactJsonContent != null, "Mule Artifact Json content cannot be null");

    this.originFolder = originFolder;
    this.applicationPomGenerator = applicationPomGenerator;
    this.pomPropertiesContent = pomPropertiesContent;
    this.muleArtifactJson = muleArtifactJsonContent;
    this.groupId = groupId;
    this.artifactId = artifactId;
    this.tempDir = createTempDirectory(null).toFile();
    this.archiver = new MuleExplodedArchiver();
  }

  public ExtensionApplicationStructureGenerator withSourcesFolder(File sourcesFolder) {
    this.sourcesFolder = sourcesFolder;
    return this;
  }

  public ExtensionApplicationStructureGenerator withArchiver(MuleArchiver archiver) {
    this.archiver = archiver;
    return this;
  }

  public ExtensionApplicationStructureGenerator withRepositoryGenerator(ApplicationRepositoryGenerator repositoryGenerator) {
    this.repositoryGenerator = repositoryGenerator;
    return this;
  }

  @Override
  public Path generate(Path destinationFolder, RunConfiguration runConfiguration) throws Exception {
    checkArgument(destinationFolder != null, "Destination folder cannot be null");

    Model pomModel = applicationPomGenerator.generate(runConfiguration);
    byte[] pomXmlContent = generatePomXml(pomModel);
    File pomFile = addPomDescriptors(pomXmlContent);
    File muleArtifact = new File(tempDir, MULE_ARTIFACT.value());
    addResources();

    if (nonNull(sourcesFolder)) {
      attachSources(pomXmlContent);
    }

    if (nonNull(repositoryGenerator)) {
      MavenProject applicationMavenProject = new MavenProject(pomModel);
      applicationMavenProject.setFile(pomFile);
      repositoryGenerator.generateRepository(applicationMavenProject, muleArtifact, tempDir);
      archiver.addRepository(new File(tempDir, REPOSITORY.value()), null, null);
    }

    addMuleArtifactJson(muleArtifact);
    archiver.setDestFile(destinationFolder.toFile());
    archiver.createArchive();

    if (nonNull(repositoryGenerator)) {
      addRunConfiguration(destinationFolder, runConfiguration);
    }

    FileUtils.cleanDirectory(tempDir);
    return destinationFolder;
  }

  private void addMuleArtifactJson(File muleArtifactFolder) throws IOException {
    FileUtils.copyToFile(new ByteArrayInputStream(muleArtifactJson), new File(muleArtifactFolder, MULE_ARTIFACT_JSON));

    archiver.addMuleArtifact(muleArtifactFolder, null, null);
  }

  private void addRunConfiguration(Path destinationFolder, RunConfiguration runConfiguration) throws IOException {
    Gson gson = new GsonBuilder().setPrettyPrinting().create();
    writeStringToFile(destinationFolder.resolve(META_INF.value()).resolve(MUNIT.value()).resolve(RUN_CONFIGURATION_JSON).toFile(),
                      gson.toJson(runConfiguration), defaultCharset());
  }

  private File addPomDescriptors(byte[] pomXmlContent) throws IOException {
    File mavenFolder = new File(tempDir, MAVEN.value());
    File groupIdFolder = new File(mavenFolder, groupId);
    File artifactIdFolder = new File(groupIdFolder, artifactId);
    File pomXmlLocation = new File(artifactIdFolder, POM_XML);
    File pomPropertiesFile = new File(artifactIdFolder, POM_PROPERTIES);

    FileUtils.copyToFile(new ByteArrayInputStream(pomXmlContent), pomXmlLocation);
    FileUtils.copyToFile(new ByteArrayInputStream(pomPropertiesContent), pomPropertiesFile);

    archiver.addMaven(mavenFolder, null, null);

    return pomXmlLocation;
  }

  protected byte[] generatePomXml(Model pomModel) throws IOException {
    ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
    new MavenXpp3Writer().write(byteArrayOutputStream, pomModel);
    return byteArrayOutputStream.toByteArray();
  }

  private void attachSources(byte[] pomXmlContent) throws IOException {
    File muleSources = new File(tempDir, MULE_SRC.value());
    File appSources = new File(muleSources, sourcesFolder.getName());
    FileUtils.copyDirectory(sourcesFolder, appSources, path -> !(path.equals(new File(sourcesFolder, TARGET.value()))));

    FileUtils.copyToFile(new ByteArrayInputStream(pomXmlContent), new File(appSources, POM_XML));
    FileUtils.copyToFile(new ByteArrayInputStream(muleArtifactJson), new File(appSources, MULE_ARTIFACT_JSON));
    String srcMainExclusion = Paths.get(sourcesFolder.getName()).resolve(SRC.value()).resolve(MAIN.value()).toString();
    archiver.addMuleSrc(muleSources, null, new String[] {srcMainExclusion + "/**"});
  }

  private void addResources() {
    addToRootIfExists(originFolder.resolve(TEST_MULE.value()).resolve(MUNIT.value()).toFile(), null, null);
    addToRootIfExists(originFolder.resolve(TEST_CLASSES.value()).toFile(), null, null);
  }

  private void addToRootIfExists(File file, String[] includes, String[] excludes) {
    if (file.exists()) {
      archiver.addToRoot(file, includes, excludes);
    }
  }
}
