/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.assertion.api.expression;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * Class that represents the result of a comparison
 *
 * @author Mulesoft Inc.
 * @since 2.2.0
 */
public class MatcherResult {

  public static Optional<MatcherResult> validateAndCreate(Map<String, Object> data) {
    if (data == null || !Stream.of("matches", "description").allMatch(data::containsKey)) {
      return Optional.empty();
    }

    return Optional.of(new MatcherResult(data));
  }

  private boolean matches;
  private Description description;
  private List<String> reasons;

  public MatcherResult() {}

  private MatcherResult(Map<String, Object> data) {
    this.matches = Boolean.TRUE.equals(data.get("matches"));
    this.description = new Description(data.get("description"));
    this.reasons = Optional.ofNullable(data.get("reasons"))
        .filter(List.class::isInstance)
        .map(values -> ((List<?>) values).stream()
            .filter(Objects::nonNull)
            .map(String::valueOf)
            .collect(Collectors.toList()))
        .orElseGet(Collections::emptyList);
  }

  public boolean isMatches() {
    return matches;
  }

  public void setMatches(boolean matches) {
    this.matches = matches;
  }

  public void setDescription(Description description) {
    this.description = description;
  }

  public void setReasons(List<String> reasons) {
    this.reasons = reasons;
  }

  public Description getDescription() {
    return description;
  }

  public List<String> getReasons() {
    return reasons;
  }

  /**
   * Description of a failed comparison
   *
   * @author Mulesoft Inc.
   * @since 2.2.0
   */
  public static class Description {

    private static String getStringOrNull(Object value) {
      return value instanceof String ? (String) value : null;
    }

    private String expected;
    private String actual;
    private String message;

    public Description() {}

    @SuppressWarnings("unchecked")
    public Description(Object data) {
      if (data instanceof Map) {
        Map<Object, Object> dataMap = (Map<Object, Object>) data;
        this.expected = getStringOrNull(dataMap.get("expected"));
        this.actual = getStringOrNull(dataMap.get("actual"));
      } else {
        this.message = getStringOrNull(data);
      }
    }

    public String getExpected() {
      return expected;
    }

    public void setExpected(String expected) {
      this.expected = expected;
    }

    public String getActual() {
      return actual;
    }

    public void setActual(String actual) {
      this.actual = actual;
    }

    public String getMessage() {
      return message;
    }

    public void setMessage(String message) {
      this.message = message;
    }

    @Override
    public String toString() {
      return "Description{" +
          "expected='" + expected + '\'' +
          ", actual='" + actual + '\'' +
          ", message='" + message + '\'' +
          '}';
    }
  }

  @Override
  public String toString() {
    return "MatcherResult{" +
        "matches=" + matches +
        ", description=" + description +
        ", reasons=" + reasons +
        '}';
  }
}
