/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.assertion.internal.matchers;

import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;

import java.util.List;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.hamcrest.BaseMatcher;
import org.hamcrest.Description;
import org.hamcrest.Matcher;

import org.mule.munit.assertion.api.matchers.Diff;
import org.mule.runtime.api.metadata.TypedValue;

/**
 * Hamcrest matcher that uses DW callbacks to perform validations
 * 
 * @since 2.1.0
 * @author Mulesoft Inc.
 */
public class CallbackMatcher extends TypedValueMatcher {

  private Function<List<Object>, Diff> callback;
  private Diff result;

  public CallbackMatcher(Function<List<Object>, Diff> callback) {
    this.callback = callback;
  }

  @Override
  public boolean doMatch(TypedValue typedValue) {
    result = callback.apply(singletonList(typedValue));
    return result.getMatches();
  }

  @Override
  public void describeTo(Description description) {
    assumeNullIfResultNotPresent();
    description.appendText(messageToString(result.getDiffs().stream().map(Diff.Difference::getExpected).collect(toList())));
  }


  @Override
  public void describeMismatch(Object item, Description description) {
    assumeNullIfResultNotPresent();
    description
        .appendText(messageToString(result.getDiffs().stream().map(Diff.Difference::getActualWithLocation).collect(toList())));
  }

  private void assumeNullIfResultNotPresent() {
    if (result == null) {
      doMatch(TypedValue.of(null));
    }
  }

  @Override
  public Matcher<Object> toPlainValueMatcher() {
    return new BaseMatcher<Object>() {

      @Override
      public boolean matches(Object o) {
        return CallbackMatcher.this.doMatch(TypedValue.of(o));
      }

      @Override
      public void describeTo(Description description) {
        CallbackMatcher.this.describeTo(description);
      }

      @Override
      public void describeMismatch(Object item, Description description) {
        CallbackMatcher.this.describeMismatch(item, description);
      }
    };
  }

  private String messageToString(List<String> messages) {
    return messages.stream().collect(Collectors.joining(","));
  }

}
