/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.assertion.internal;


import static java.util.Objects.nonNull;

import java.util.List;
import java.util.stream.Collectors;

import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.mule.munit.assertion.api.expression.MatcherResult;
import org.mule.munit.assertion.internal.matchers.ValueMatcher;
import org.mule.runtime.api.metadata.TypedValue;



/**
 * <p>
 * Module to assert payload's results
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.0.2
 */
public class AssertModule {

  /**
   * <p>
   * Assert that expression matches a condition .
   * </p>
   * <p/>
   *
   * @param message Description message to be shown in case of failure.
   * @param expression Expression result to be asserted
   * @param is Matcher expression
   */
  public void assertThat(String message, TypedValue expression, Matcher is) {
    MatcherAssert.assertThat(wrapMessage(message), expression, is);
  }

  /**
   * <p>
   * Assert that an expression value matches a condition .
   * </p>
   * <p>
   * Useful for testing since it is easier to use
   * </p>
   * <p/>
   *
   * @param message Description message to be shown in case of failure.
   * @param value Expression value to be asserted
   * @param is Matcher expression
   */
  protected void assertThat(String message, Object value, Matcher is) {
    TypedValue typedValue = new TypedValue<>(value, null);
    Matcher<TypedValue> typedValueMatcher = new ValueMatcher(is);
    assertThat(message, typedValue, typedValueMatcher);
  }

  /**
   * <p>
   * Asserts the results of a matcher expression
   * </p>
   * <p/>
   *
   * @param message Description message to be shown in case of failure.
   * @param matcherResult Matcher result to be asserted
   */
  public void assertMatcherResult(MatcherResult matcherResult, String message) {
    if (!matcherResult.isMatches()) {
      throw new AssertionError(generateMatcherResultMessage(matcherResult, message));
    }
  }

  private static String wrapMessage(String message) {
    return message == null ? "" : message;
  }

  private static String generateMatcherResultMessage(MatcherResult matcherResult, String message) {
    MatcherResult.Description description = matcherResult.getDescription();
    List<String> reasons = matcherResult.getReasons();
    StringBuilder messageBuilder = new StringBuilder();
    messageBuilder.append(message);
    messageBuilder.append(System.lineSeparator()).append("Expected: ").append(description.getExpected());
    messageBuilder.append(System.lineSeparator()).append("Actual: ").append(description.getActual());
    if (nonNull(reasons) && !reasons.isEmpty()) {
      messageBuilder.append(System.lineSeparator()).append("Reasons: ");
      messageBuilder.append(reasons.stream().collect(Collectors.joining(", ")));
    }

    return messageBuilder.toString();
  }
}
